@extends('layouts/layoutMaster')

@section('title', 'Chain Select Component Examples')

<!-- Vendor Styles -->
@section('vendor-style')
@vite(['resources/assets/vendor/libs/select2/select2.scss'])
@endsection

<!-- Vendor Scripts -->
@section('vendor-script')
@vite(['resources/assets/vendor/libs/select2/select2.js'])
@endsection

<!-- Page Scripts -->
@section('page-script')
@vite(['resources/assets/js/chain-select.js'])
<script type="module">
  'use strict';

  // Wait for all modules to load
  window.addEventListener('load', function() {
    console.log('[DEBUG] Window loaded');
    console.log('[DEBUG] jQuery available:', typeof $ !== 'undefined');
    console.log('[DEBUG] Select2 available:', typeof $.fn.select2 !== 'undefined');

    if (typeof $ === 'undefined' || typeof $.fn.select2 === 'undefined') {
      console.error('[ERROR] jQuery or Select2 is not loaded!');
      return;
    }

    var select2Elements = $('.select2');
    console.log('[DEBUG] Found', select2Elements.length, 'select2 elements');

    if (select2Elements.length) {
      select2Elements.each(function () {
        var $this = $(this);

        // Don't re-initialize
        if ($this.hasClass('select2-hidden-accessible')) {
          return;
        }

        console.log('[DEBUG] Initializing:', $this.attr('name'));

        // Use select2Focus helper if available
        if (typeof select2Focus !== 'undefined') {
          select2Focus($this);
        }

        // Check if AJAX search is enabled
        var isAjaxSearch = $this.data('ajax-search') === true;
        var dependsOn = $this.data('depends-on');
        var filterColumn = $this.data('filter-column');

        var select2Config = {
          placeholder: $this.data('placeholder') || 'Select value',
          dropdownParent: $this.parent()
        };

        // Configure AJAX if enabled
        if (isAjaxSearch) {
          var ajaxModel = $this.data('ajax-model');
          var ajaxDisplay = $this.data('ajax-display');
          var ajaxKey = $this.data('ajax-key');
          var ajaxLimit = $this.data('ajax-limit') || 20;

          console.log('[DEBUG] AJAX Search enabled for:', $this.attr('name'));

          select2Config.ajax = {
            url: '/api/ajax-search',
            dataType: 'json',
            delay: 250,
            transport: function(params, success, failure) {
              // Check if there's a search term
              var data = params.data;
              if (!data || !data.search || data.search.length < 1) {
                console.log('[DEBUG] No search term, returning empty results');
                // Return empty results immediately without making request
                success({
                  results: [],
                  pagination: { more: false }
                });
                return;
              }

              // Make the actual AJAX request
              var $request = $.ajax(params);

              $request.then(success);
              $request.fail(failure);

              return $request;
            },
            data: function (params) {
              console.log('[DEBUG] AJAX data function called with term:', params.term);

              var data = {
                search: params.term || '',
                page: params.page || 1,
                model: ajaxModel,
                display: ajaxDisplay,
                key: ajaxKey,
                limit: ajaxLimit
              };

              // If this select depends on another, include parent filter
              if (dependsOn && filterColumn) {
                var parentSelect = document.querySelector('select[name="' + dependsOn + '"], select[name="' + dependsOn + '[]"]');
                if (parentSelect) {
                  var parentValue = parentSelect.multiple
                    ? Array.from(parentSelect.selectedOptions).map(opt => opt.value).filter(v => v !== '')
                    : parentSelect.value;

                  if (parentValue && (Array.isArray(parentValue) ? parentValue.length > 0 : parentValue !== '')) {
                    data.filterColumn = filterColumn;
                    data.filterValue = parentValue;
                    console.log('[DEBUG] Parent filter applied:', filterColumn, '=', parentValue);
                  }
                }
              }

              return data;
            },
            processResults: function (data, params) {
              params.page = params.page || 1;

              // Return empty if no data (prevents showing anything when no search term)
              if (!data || !data.results) {
                return {
                  results: [],
                  pagination: { more: false }
                };
              }

              return {
                results: data.results,
                pagination: {
                  more: data.pagination.more
                }
              };
            },
            cache: false // Disable cache to ensure fresh searches
          };
          select2Config.minimumInputLength = 1; // Require at least 1 character to search
          select2Config.placeholder = $this.data('placeholder') || 'Type to search...';
          select2Config.allowClear = true; // Allow clearing selection
        }

        // Wrap and initialize select2
        $this.wrap('<div class="position-relative"></div>').select2(select2Config);

        console.log('[DEBUG] Successfully initialized:', $this.attr('name'));
      });
    }

    console.log('[DEBUG] Select2 initialization complete');
  });
</script>
@endsection

@section('content')
    <div class="container">
        <div class="row">
            <div class="col-12">
                <div class="card mb-4">
                    <div class="card-header">
                        <h4 class="card-title mb-0">Chain Selection - Unlimited Dependent Dropdowns</h4>
                        <p class="text-muted mt-2 mb-0">Select dropdowns automatically populate based on parent selection. Works for unlimited levels.</p>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="#">
                            @csrf

                            <div class="row">
                                <!-- Level 1: Province (Independent) -->
                                <div class="col-md-4 mb-4">
                                    <label for="province_id" class="form-label">Province</label>
                                    <x-form.select
                                        name="province_id"
                                        model="App\Models\CdProvince"
                                        {{-- :query="[['orderBy',['name','asc']]]" --}}
                                        display="name"
                                        key="id"
                                        placeholder="Select Province"
                                        class="form-control select2"
                                        id="province_id"
                                        data-placeholder="Select Province"
                                    />
                                </div>

                                <!-- Level 2: District (Depends on Province) -->
                                <div class="col-md-4 mb-4">
                                    <label for="district_id" class="form-label">District</label>
                                    <x-form.select
                                        name="district_id"
                                        model="App\Models\CdDistrict"
                                        display="name"
                                        key="id"
                                        placeholder="Select District"
                                        class="form-control select2"
                                        id="district_id"
                                        dependsOn="province_id"
                                        filterColumn="cdprovince_id"
                                        data-placeholder="Select District"
                                    />
                                </div>

                                <!-- Level 3: City (Depends on District) -->
                                <div class="col-md-4 mb-4">
                                    <label for="city_id" class="form-label">City</label>
                                    <x-form.select
                                        name="city_id"
                                        model="App\Models\CdCity"
                                        display="name"
                                        key="id"
                                        placeholder="Select City"
                                        class="form-control select2"
                                        id="city_id"
                                        dependsOn="district_id"
                                        filterColumn="cddistrict_id"
                                        data-placeholder="Select City"
                                    />
                                </div>
                            </div>

                            <div class="alert alert-info">
                                <strong>How it works:</strong>
                                <ul class="mb-0 mt-2">
                                    <li>Select a <strong>Province</strong> first</li>
                                    <li>The <strong>District</strong> dropdown will populate with districts from that province</li>
                                    <li>Select a district to populate the <strong>City</strong> dropdown</li>
                                    <li>You can chain as many levels as you need!</li>
                                </ul>
                            </div>

                            <button type="submit" class="btn btn-primary">Submit</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-12">
                <div class="card mb-4">
                    <div class="card-header">
                        <h4 class="card-title mb-0">Multiple Selection - 3-Level Chain (Province → District → City)</h4>
                        <p class="text-muted mt-2 mb-0">Select multiple provinces → all their districts → all their cities. Complete cascading example.</p>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="#">
                            @csrf

                            <div class="row">
                                <!-- Level 1: Multiple Provinces -->
                                <div class="col-md-4 mb-4">
                                    <label for="multi_province_ids" class="form-label">Provinces (Multiple)</label>
                                    <x-form.select
                                        name="multi_province_ids"
                                        model="App\Models\CdProvince"
                                        :query="[['orderBy',['name','asc']]]"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="multi_province_ids"
                                        :multiple="true"
                                        data-placeholder="Select Provinces"
                                    />
                                    <small class="text-muted">Hold Ctrl/Cmd to select multiple provinces</small>
                                </div>

                                <!-- Level 2: Districts from Multiple Provinces -->
                                <div class="col-md-4 mb-4">
                                    <label for="multi_district_ids" class="form-label">Districts (All from selected provinces)</label>
                                    <x-form.select
                                        name="multi_district_ids"
                                        model="App\Models\CdDistrict"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="multi_district_ids"
                                        dependsOn="multi_province_ids"
                                        filterColumn="cdprovince_id"
                                        :multiple="true"
                                        data-placeholder="Select Districts"
                                    />
                                    <small class="text-muted">Shows all districts from selected provinces</small>
                                </div>

                                <!-- Level 3: Cities from Multiple Districts -->
                                <div class="col-md-4 mb-4">
                                    <label for="multi_city_ids" class="form-label">Cities (All from selected districts)</label>
                                    <x-form.select
                                        name="multi_city_ids"
                                        model="App\Models\CdCity"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="multi_city_ids"
                                        dependsOn="multi_district_ids"
                                        filterColumn="cddistrict_id"
                                        :multiple="true"
                                        data-placeholder="Select Cities"
                                    />
                                    <small class="text-muted">Shows all cities from selected districts</small>
                                </div>
                            </div>

                            <div class="alert alert-success">
                                <strong>3-Level Multiple Selection:</strong>
                                <ul class="mb-0 mt-2">
                                    <li>Select <strong>multiple provinces</strong> (e.g., Punjab, Sindh)</li>
                                    <li>The <strong>Districts</strong> dropdown shows ALL districts from selected provinces</li>
                                    <li>Select <strong>multiple districts</strong> (e.g., Lahore, Karachi, Multan)</li>
                                    <li>The <strong>Cities</strong> dropdown shows ALL cities from selected districts</li>
                                    <li>Select as many cities as you need!</li>
                                </ul>
                            </div>

                            <button type="submit" class="btn btn-primary">Submit</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-12">
                <div class="card mb-4">
                    <div class="card-header">
                        <h4 class="card-title mb-0">Multiple Selection - 3-Level Edit Mode (Province → District → City)</h4>
                        <p class="text-muted mt-2 mb-0">Pre-selected multiple values across 3 levels for editing.</p>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="#">
                            @csrf

                            @php
                                // Simulate editing with multiple selections across 3 levels
                                $selectedProvinces = [1, 3, 5];      // Multiple provinces
                                $selectedDistricts = [2, 7, 12];     // Multiple districts from those provinces
                                $selectedCities = [10, 25, 45, 78];  // Multiple cities from those districts
                            @endphp

                            <div class="row">
                                <!-- Multiple Provinces Pre-selected -->
                                <div class="col-md-4 mb-4">
                                    <label for="edit_3lvl_province_ids" class="form-label">Provinces (Pre-selected)</label>
                                    <x-form.select
                                        name="edit_3lvl_province_ids"
                                        model="App\Models\CdProvince"
                                        :query="[['orderBy',['name','asc']]]"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="edit_3lvl_province_ids"
                                        :multiple="true"
                                        :value="$selectedProvinces"
                                        data-placeholder="Select Provinces"
                                    />
                                    <small class="text-muted">Values: {{ implode(', ', $selectedProvinces) }}</small>
                                </div>

                                <!-- Multiple Districts Pre-selected -->
                                <div class="col-md-4 mb-4">
                                    <label for="edit_3lvl_district_ids" class="form-label">Districts (Pre-selected)</label>
                                    <x-form.select
                                        name="edit_3lvl_district_ids"
                                        model="App\Models\CdDistrict"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="edit_3lvl_district_ids"
                                        dependsOn="edit_3lvl_province_ids"
                                        filterColumn="cdprovince_id"
                                        :multiple="true"
                                        :value="$selectedDistricts"
                                        data-placeholder="Select Districts"
                                    />
                                    <small class="text-muted">Values: {{ implode(', ', $selectedDistricts) }}</small>
                                </div>

                                <!-- Multiple Cities Pre-selected -->
                                <div class="col-md-4 mb-4">
                                    <label for="edit_3lvl_city_ids" class="form-label">Cities (Pre-selected)</label>
                                    <x-form.select
                                        name="edit_3lvl_city_ids"
                                        model="App\Models\CdCity"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="edit_3lvl_city_ids"
                                        dependsOn="edit_3lvl_district_ids"
                                        filterColumn="cddistrict_id"
                                        :multiple="true"
                                        :value="$selectedCities"
                                        data-placeholder="Select Cities"
                                    />
                                    <small class="text-muted">Values: {{ implode(', ', $selectedCities) }}</small>
                                </div>
                            </div>

                            <div class="alert alert-info">
                                <strong>3-Level Edit Mode:</strong>
                                <ul class="mb-0 mt-2">
                                    <li>All 3 levels load with <strong>pre-selected multiple values</strong></li>
                                    <li>Provinces load first → Districts from those provinces → Cities from those districts</li>
                                    <li>Everything cascades automatically on page load</li>
                                    <li>User can modify any level and children update accordingly</li>
                                    <li>Perfect for complex edit forms with multi-select relationships</li>
                                </ul>
                            </div>

                            <button type="submit" class="btn btn-primary">Update</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-12">
                <div class="card mb-4">
                    <div class="card-header">
                        <h4 class="card-title mb-0">Edit Mode - Pre-selected Values (Single)</h4>
                        <p class="text-muted mt-2 mb-0">When editing existing records, pass the value attribute to pre-select options.</p>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="#">
                            @csrf

                            @php
                                // Simulate editing a record with existing values
                                $existingProvinceId = 7;  // Change this to test different provinces
                                $existingDistrictId = 2;  // Change this to test different districts
                                $existingCityId = 492;     // Change this to test different cities
                            @endphp

                            <div class="row">
                                <!-- Level 1: Province (with pre-selected value) -->
                                <div class="col-md-4 mb-4">
                                    <label for="edit_province_id" class="form-label">Province (Pre-selected)</label>
                                    <x-form.select
                                        name="edit_province_id"
                                        model="App\Models\CdProvince"
                                        :query="[['orderBy',['name','asc']]]"
                                        display="name"
                                        key="id"
                                        placeholder="Select Province"
                                        class="form-control select2"
                                        id="edit_province_id"
                                        :value="$existingProvinceId"
                                        data-placeholder="Select Province"
                                    />
                                    <small class="text-muted">Value: {{ $existingProvinceId }}</small>
                                </div>

                                <!-- Level 2: District (with pre-selected value) -->
                                <div class="col-md-4 mb-4">
                                    <label for="edit_district_id" class="form-label">District (Pre-selected)</label>
                                    <x-form.select
                                        name="edit_district_id"
                                        model="App\Models\CdDistrict"
                                        display="name"
                                        key="id"
                                        placeholder="Select District"
                                        class="form-control select2"
                                        id="edit_district_id"
                                        dependsOn="edit_province_id"
                                        filterColumn="cdprovince_id"
                                        :value="$existingDistrictId"
                                        data-placeholder="Select District"
                                    />
                                    <small class="text-muted">Value: {{ $existingDistrictId }}</small>
                                </div>

                                <!-- Level 3: City (with pre-selected value) -->
                                <div class="col-md-4 mb-4">
                                    <label for="edit_city_id" class="form-label">City (Pre-selected)</label>
                                    <x-form.select
                                        name="edit_city_id"
                                        model="App\Models\CdCity"
                                        display="name"
                                        key="id"
                                        placeholder="Select City"
                                        class="form-control select2"
                                        id="edit_city_id"
                                        dependsOn="edit_district_id"
                                        filterColumn="cddistrict_id"
                                        :value="$existingCityId"
                                        data-placeholder="Select City"
                                    />
                                    <small class="text-muted">Value: {{ $existingCityId }}</small>
                                </div>
                            </div>

                            <div class="alert alert-success">
                                <strong>Edit Mode Features:</strong>
                                <ul class="mb-0 mt-2">
                                    <li>All dropdowns load with pre-selected values automatically</li>
                                    <li>Dependent dropdowns populate with correct options on page load</li>
                                    <li>User can change any level and children will update accordingly</li>
                                    <li>Perfect for edit forms where you're modifying existing data</li>
                                </ul>
                            </div>

                            <button type="submit" class="btn btn-primary">Update</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-12">
                <div class="card mb-4">
                    <div class="card-header">
                        <h4 class="card-title mb-0">Standard Select (Non-Dependent)</h4>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="#">
                            @csrf

                            <h5 class="mb-3">Using a Model with Query</h5>
                            <div class="mb-4">
                                <label for="province_standard" class="form-label">Province</label>
                                <x-form.select
                                    name="province_standard"
                                    model="App\Models\CdProvince"
                                    :query="[['orderBy',['name','asc']]]"
                                    display="name"
                                    key="id"
                                    placeholder="Select Province"
                                    class="form-control select2"
                                    id="province_standard"
                                    data-placeholder="Select Province"
                                />
                            </div>

                            <h5 class="mb-3">Passing Items Directly</h5>
                            @php
                                $countries = [
                                    (object)['code' => 'US', 'label' => 'United States'],
                                    (object)['code' => 'CA', 'label' => 'Canada'],
                                    (object)['code' => 'GB', 'label' => 'United Kingdom'],
                                ];
                            @endphp

                            <div class="mb-4">
                                <label for="country" class="form-label">Country</label>
                                <x-form.select
                                    name="country"
                                    :items="$countries"
                                    display="label"
                                    key="code"
                                    placeholder="Choose a country"
                                    class="form-control select2"
                                    id="country"
                                    data-placeholder="Choose a country"
                                />
                            </div>

                            <button type="submit" class="btn btn-primary">Submit</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-12">
                <div class="card mb-4">
                    <div class="card-header bg-success">
                        <h4 class="card-title mb-0 text-white">AJAX Search - Large Datasets (300+ Records)</h4>
                        <p class="text-white mt-2 mb-0">Search records dynamically from server instead of loading all at once. Perfect for large datasets!</p>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="#">
                            @csrf

                            <div class="row">
                                <!-- AJAX Search - Province -->
                                <div class="col-md-4 mb-4">
                                    <label for="ajax_province_id" class="form-label">Province (AJAX Search)</label>
                                    <x-form.select
                                        name="ajax_province_id"
                                        model="App\Models\CdProvince"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="ajax_province_id"
                                        :ajax="true"
                                        :ajaxLimit="20"
                                        data-placeholder="Type to search provinces..."
                                    />
                                    <small class="text-muted">Only loads 20 records at a time, searches as you type</small>
                                </div>

                                <!-- AJAX Search - District (Chain with AJAX parent) -->
                                <div class="col-md-4 mb-4">
                                    <label for="ajax_district_id" class="form-label">District (Depends on AJAX Province)</label>
                                    <x-form.select
                                        name="ajax_district_id"
                                        model="App\Models\CdDistrict"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="ajax_district_id"
                                        dependsOn="ajax_province_id"
                                        filterColumn="cdprovince_id"
                                        :ajax="false"
                                        :ajaxLimit="15"
                                        data-placeholder="Type to search districts..."
                                    />
                                    <small class="text-muted">Type to search districts from selected province (limit 15)</small>
                                </div>

                                <!-- AJAX Search - City -->
                                <div class="col-md-4 mb-4">
                                    <label for="ajax_city_id" class="form-label">City (AJAX with limit 10)</label>
                                    <x-form.select
                                        name="ajax_city_id"
                                        model="App\Models\CdCity"
                                        display="name"
                                        key="id"
                                        class="form-control select2"
                                        id="ajax_city_id"
                                        dependsOn="ajax_district_id"
                                        filterColumn="cddistrict_id"
                                        :ajax="false"
                                        :ajaxLimit="10"
                                        data-placeholder="Type to search cities..."
                                    />
                                    <small class="text-muted">Type to search cities from selected district (limit 10)</small>
                                </div>
                            </div>

                            <div class="alert alert-success">
                                <strong>AJAX Search Benefits:</strong>
                                <ul class="mb-0 mt-2">
                                    <li>⚡ <strong>Fast Performance</strong> - Only loads records when needed</li>
                                    <li>🔍 <strong>Real-time Search</strong> - Searches database as you type</li>
                                    <li>📊 <strong>Handles Large Datasets</strong> - Works with thousands of records</li>
                                    <li>🎯 <strong>Customizable Limit</strong> - Control how many results to show</li>
                                    <li>🔗 <strong>Works with Chain Selection</strong> - Combine with dependent dropdowns</li>
                                    <li>💾 <strong>Pagination Support</strong> - Load more results on scroll</li>
                                </ul>
                            </div>

                            <h5 class="mt-4">AJAX Search Syntax:</h5>
                            <pre class="bg-light p-3 rounded"><code>{{-- Enable AJAX search with custom limit --}}
&lt;x-form.select
    name="province_id"
    model="App\Models\CdProvince"
    display="name"
    key="id"
    class="form-control select2"
    :ajax="true"
    :ajaxLimit="20"
    data-placeholder="Type to search..."
/&gt;

{{-- Combine AJAX with chain selection --}}
&lt;x-form.select
    name="district_id"
    model="App\Models\CdDistrict"
    display="name"
    key="id"
    dependsOn="province_id"
    filterColumn="cdprovince_id"
    class="form-control select2"
    :ajax="true"
    :ajaxLimit="15"
/&gt;</code></pre>

                            <button type="submit" class="btn btn-success">Submit</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-12">
                <div class="card mb-4">
                    <div class="card-header">
                        <h4 class="card-title mb-0">Usage Documentation</h4>
                    </div>
                    <div class="card-body">
                        <h5>Chain Selection Syntax:</h5>
                        <pre class="bg-light p-3 rounded"><code>{{-- Single Selection --}}
&lt;x-form.select
    name="district_id"
    model="App\Models\CdDistrict"
    display="name"
    key="id"
    placeholder="Select District"
    dependsOn="province_id"
    filterColumn="cdprovince_id"
/&gt;

{{-- Multiple Selection --}}
&lt;x-form.select
    name="district_ids"
    model="App\Models\CdDistrict"
    display="name"
    key="id"
    dependsOn="province_ids"
    filterColumn="cdprovince_id"
    :multiple="true"
/&gt;</code></pre>

                        <h5 class="mt-4">For Edit Forms (Pre-selected Values):</h5>
                        <pre class="bg-light p-3 rounded"><code>{{-- Single Selection --}}
&lt;x-form.select
    name="province_id"
    model="App\Models\CdProvince"
    display="name"
    key="id"
    :value="$record->province_id"
/&gt;

&lt;x-form.select
    name="district_id"
    model="App\Models\CdDistrict"
    display="name"
    key="id"
    dependsOn="province_id"
    filterColumn="cdprovince_id"
    :value="$record->district_id"
/&gt;

{{-- Multiple Selection --}}
&lt;x-form.select
    name="province_ids"
    model="App\Models\CdProvince"
    display="name"
    key="id"
    :multiple="true"
    :value="$record->province_ids"  {{-- Array: [1, 3, 5] --}}
/&gt;

&lt;x-form.select
    name="district_ids"
    model="App\Models\CdDistrict"
    display="name"
    key="id"
    dependsOn="province_ids"
    filterColumn="cdprovince_id"
    :multiple="true"
    :value="$record->district_ids"  {{-- Array: [2, 7, 12] --}}
/&gt;</code></pre>

                        <h5 class="mt-4">Key Attributes for Chain Selection:</h5>
                        <ul>
                            <li><strong>dependsOn:</strong> The name of the parent select element</li>
                            <li><strong>filterColumn:</strong> The database column to filter by (usually a foreign key)</li>
                            <li><strong>model:</strong> The Laravel model class to query</li>
                            <li><strong>display:</strong> The column to display in options</li>
                            <li><strong>key:</strong> The column to use as option value</li>
                            <li><strong>:multiple="true":</strong> Enable multiple selection</li>
                            <li><strong>:value:</strong> Pre-selected value(s) - single value or array for multiple</li>
                            <li><strong>:ajax="true":</strong> Enable AJAX search for large datasets</li>
                            <li><strong>:ajaxLimit="20":</strong> Limit number of results per AJAX search (default: 20)</li>
                        </ul>

                        <h5 class="mt-4">Features:</h5>
                        <ul>
                            <li>✅ Unlimited chain levels (Level 1 → Level 2 → Level 3 → ...)</li>
                            <li>✅ Automatic AJAX loading of dependent options</li>
                            <li>✅ Auto-reset child dropdowns when parent changes</li>
                            <li>✅ Works with any Laravel model</li>
                            <li>✅ No hardcoded logic - completely generic</li>
                            <li>✅ <strong>Edit mode support - pre-select values automatically</strong></li>
                            <li>✅ Works with old() helper for validation errors</li>
                            <li>✅ <strong>Multiple selection support - select multiple parents, get all related children</strong></li>
                            <li>✅ <strong>Multiple edit mode - pre-select multiple values</strong></li>
                            <li>✅ <strong>Select2 integration - beautiful searchable dropdowns with tags</strong></li>
                            <li>✅ <strong>AJAX search - handle large datasets (300+ records) efficiently</strong></li>
                            <li>✅ <strong>Customizable result limits - control performance and UX</strong></li>
                        </ul>

                        <h5 class="mt-4">Using with Select2:</h5>
                        <p>Add the <code>select2</code> class to any select element to enable Select2 features:</p>
                        <ul>
                            <li>✅ Searchable dropdowns with live filtering</li>
                            <li>✅ Beautiful tag-style multiple selections</li>
                            <li>✅ Clear button to reset selection</li>
                            <li>✅ Fully compatible with chain selection</li>
                            <li>✅ Works in both create and edit modes</li>
                        </ul>
                        <pre class="bg-light p-3 rounded"><code>&lt;x-form.select
    name="province_id"
    model="App\Models\CdProvince"
    class="form-control select2"
    data-placeholder="Search provinces..."
/&gt;</code></pre>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection
