/**
 * DataTables for Staff Management
 */

'use strict';

let dt_staff;

// Global function to reload staff table
window.reloadStaffTable = function () {
  if (typeof dt_staff !== 'undefined' && dt_staff) {
    dt_staff.ajax.reload(null, false);
    return true;
  }
  return false;
};

// Open staff form modal (create new)
window.openStaffForm = function () {
  loadStaffForm('/admin/staff/create');
};

// Edit staff (load existing)
window.editStaff = function (id) {
  loadStaffForm(`/admin/staff/edit/${id}`);
};

// Common function to load staff form
function loadStaffForm(url) {
  const staffModalElement = document.getElementById('staffModal');

  fetch(url, {
    headers: {
      'X-Requested-With': 'XMLHttpRequest',
      Accept: 'text/html'
    }
  })
    .then(response => {
      if (!response.ok) {
        return response.json().then(errorData => {
          throw errorData;
        });
      }
      return response.text();
    })
    .then(html => {
      $('#staffModalContent').html(html);
      let staffModal = new bootstrap.Modal(staffModalElement);
      staffModal.show();
    })
    .catch(error => {
      console.error('Error loading form:', error);

      if (typeof Swal !== 'undefined' && error && error.error) {
        Swal.fire({
          icon: 'error',
          title: 'Error',
          text: error.error,
          confirmButtonText: 'OK',
          customClass: {
            confirmButton: 'btn btn-primary'
          }
        });
      } else {
        alert('Failed to load staff form: ' + (error.error || 'Unknown error'));
      }
    });
}

// Reset filters
window.resetFilters = function () {
  document.getElementById('typeFilter').value = '';
  document.getElementById('statusFilter').value = '';
  dt_staff.ajax.reload();
};

$(function () {
  console.log('Staff DataTables script loaded');
  console.log('jQuery version:', $.fn.jquery);
  console.log('Looking for #staffTable:', $('#staffTable').length);

  // Check if DataTables is available
  if (typeof $.fn.DataTable === 'undefined') {
    console.error('DataTables library is not loaded!');
    return;
  }

  // Staff DataTable
  try {
    dt_staff = $('#staffTable').DataTable({
      processing: true,
      serverSide: true,
      ajax: {
        url: '/admin/staff/data',
        data: function (d) {
          d.type_filter = $('#typeFilter').val();
          d.status_filter = $('#statusFilter').val();
        },
        dataSrc: function (json) {
          console.log('Server response:', json);
          console.log('Data array:', json.data);
          if (json.data && json.data.length > 0) {
            console.log('First record:', json.data[0]);
          }
          return json.data;
        },
        error: function (xhr, error, code) {
          console.error('DataTables AJAX Error:', error, code);
          console.error('XHR:', xhr);
        }
      },
      columns: [
        { data: 'name' },
        { data: 'email' },
        { data: 'mob' },
        { data: 'type' },
        {
          data: 'status',
          render: function (data) {
            if (data == 0) {
              return '<span class="badge bg-success">Active</span>';
            } else {
              return '<span class="badge bg-danger">Disabled</span>';
            }
          }
        },
        { data: 'joined_at' },
        {
          data: 'encrypted_id',
          orderable: false,
          searchable: false,
          render: function (data, type, row) {
            return `
            <div class="d-flex gap-2">
              <button type="button" class="btn btn-sm btn-icon btn-text-secondary rounded-pill waves-effect" 
                onclick="editStaff('${data}')" title="Edit">
                <i class="ri-edit-box-line ri-20px"></i>
              </button>
            </div>
          `;
          }
        }
      ],
      order: [[0, 'asc']],
      dom: '<"row mx-1"<"col-sm-12 col-md-6"l><"col-sm-12 col-md-6"f>>t<"row mx-1"<"col-sm-12 col-md-6"i><"col-sm-12 col-md-6"p>>',
      language: {
        search: '',
        searchPlaceholder: 'Search staff...',
        lengthMenu: '_MENU_',
        info: 'Showing _START_ to _END_ of _TOTAL_ staff members',
        infoEmpty: 'No staff members found',
        infoFiltered: '(filtered from _MAX_ total)',
        paginate: {
          next: '<i class="ri-arrow-right-s-line"></i>',
          previous: '<i class="ri-arrow-left-s-line"></i>'
        }
      }
    });

    console.log('DataTable initialized:', dt_staff);

    // Filter change handlers
    $('#typeFilter, #statusFilter').on('change', function () {
      dt_staff.ajax.reload();
    });
  } catch (e) {
    console.error('Error initializing DataTable:', e);
  }
});
