/**
 * DataTables for Student Fee Management - Server Side Processing
 */

'use strict';

let fv, dt_basic;

// Toast notification helper
function showStudentToast(message, type = 'success') {
  const toastElement = document.getElementById('studentToast');
  const toastTitle = document.getElementById('toastTitle');
  const toastIcon = document.getElementById('toastIcon');
  const toastMessage = document.getElementById('toastMessage');

  if (!toastElement) {
    console.error('Toast element not found');
    return;
  }

  // Set message
  toastMessage.textContent = message;

  // Set color and icon based on type
  if (type === 'success') {
    toastTitle.textContent = 'Success';
    toastIcon.className = 'ri-checkbox-circle-fill me-2 text-success';
    toastElement.classList.add('text-success');
  } else if (type === 'error') {
    toastTitle.textContent = 'Error';
    toastIcon.className = 'ri-error-warning-fill me-2 text-danger';
    toastElement.classList.add('text-danger');
  }

  // Show toast
  const toast = new bootstrap.Toast(toastElement);
  toast.show();
}

// View fee history - Load payment history in modal
function viewFeeHistory(studentId) {
  const modalElement = document.getElementById('paymentHistoryModal');

  fetch(`/admin/student/payment-history/${studentId}`, {
    headers: {
      'X-Requested-With': 'XMLHttpRequest',
      Accept: 'text/html'
    }
  })
    .then(response => response.text())
    .then(html => {
      document.getElementById('paymentHistoryModalContent').innerHTML = html;
      let modal = new bootstrap.Modal(modalElement);
      modal.show();
    })
    .catch(error => {
      console.error('Error loading payment history:', error);
      document.getElementById('paymentHistoryModalContent').innerHTML =
        '<div class="alert alert-danger m-3">Failed to load payment history</div>';
    });
}

document.addEventListener('DOMContentLoaded', function (e) {
  // No add new button event handler needed for this page
});

// DataTable (jQuery)
$(function () {
  var dt_basic_table = $('.datatables-basic');

  // Handle filter button click
  $('#filterBtn').on('click', function () {
    var classId = $('#filterClass').val();
    var sectionId = $('#filterSection').val();
    var feeStatus = $('#filterFeeStatus').val();

    console.log('Filter applied - Class:', classId, 'Section:', sectionId, 'Fee Status:', feeStatus);

    // Reload DataTable with filters
    if (dt_basic) {
      dt_basic.ajax.reload(function () {
        console.log('DataTable reloaded with filters');
      });
    }
  });

  // Handle clear filters button click
  $('#clearFilterBtn').on('click', function () {
    console.log('Clearing all filters');

    // Reset all filter dropdowns
    $('#filterClass').val('').trigger('change.select2');
    $('#filterSection').val('').trigger('change.select2');
    $('#filterFeeStatus').val('').trigger('change.select2');

    // Reload DataTable without filters
    if (dt_basic) {
      dt_basic.ajax.reload(function () {
        console.log('DataTable reloaded - showing all records');
      });
    }
  });

  // DataTable with server-side processing
  if (dt_basic_table.length) {
    dt_basic = dt_basic_table.DataTable({
      processing: true,
      serverSide: true,
      order: [[2, 'desc']], // Sort by Roll No (column 2) in descending order
      ajax: {
        url: baseUrl + 'admin/student/managefee/data',
        type: 'GET',
        data: function (d) {
          // Add filter parameters
          d.class_id = $('#filterClass').val();
          d.section_id = $('#filterSection').val();
          d.fee_status = $('#filterFeeStatus').val();
          console.log('DataTable request:', d); // Log the request data
          return d;
        }
      },
      columns: [
        { data: '' }, // Responsive control
        { data: 'id' }, // Checkbox
        { data: 'roll_no' }, // Roll No
        { data: 'name' }, // Name with image
        { data: 'fname' }, // Father Name
        { data: 'mobile' }, // Mobile
        { data: 'join_date' }, // Join Date
        { data: 'class_section' }, // Class/Section
        { data: 'fstatus' }, // Status
        { data: 'pending_fee' }, // Pending Fee
        { data: '' } // Actions
      ],
      columnDefs: [
        {
          // For Responsive
          className: 'control',
          orderable: false,
          searchable: false,
          responsivePriority: 2,
          targets: 0,
          render: function (data, type, full, meta) {
            return '';
          }
        },
        {
          // For Checkboxes
          targets: 1,
          orderable: false,
          searchable: false,
          responsivePriority: 3,
          checkboxes: true,
          render: function () {
            return '<input type="checkbox" class="dt-checkboxes form-check-input">';
          },
          checkboxes: {
            selectAllRender: '<input type="checkbox" class="form-check-input">'
          }
        },
        {
          // Roll No
          targets: 2,
          orderable: true,
          responsivePriority: 4,
          className: 'text-center',
          render: function (data, type, full, meta) {
            return '<span class="fw-medium">' + (data || 'N/A') + '</span>';
          }
        },
        {
          // Name with Avatar image
          targets: 3,
          responsivePriority: 1,
          render: function (data, type, full, meta) {
            var $user_img = full['image'];
            var $name = full['name'];

            var $output = '<img src="' + $user_img + '" alt="Avatar" class="rounded-circle" width="48" height="48">';

            // Creates full output for row
            var $row_output =
              '<div class="d-flex justify-content-start align-items-center user-name">' +
              '<div class="avatar-wrapper">' +
              '<div class="avatar me-2">' +
              $output +
              '</div>' +
              '</div>' +
              '<div class="d-flex flex-column">' +
              '<span class="emp_name text-truncate text-heading fw-medium">' +
              $name +
              '</span>' +
              '</div>' +
              '</div>';
            return $row_output;
          }
        },
        {
          // Father Name
          targets: 4,
          responsivePriority: 5,
          render: function (data, type, full, meta) {
            return data || 'N/A';
          }
        },
        {
          // Mobile Numbers
          targets: 5,
          responsivePriority: 6,
          width: '120px',
          className: 'text-center',
          render: function (data, type, full, meta) {
            return data || 'N/A';
          }
        },
        {
          // Join Date
          targets: 6,
          responsivePriority: 7,
          className: 'text-center',
          render: function (data, type, full, meta) {
            if (!data) return 'N/A';
            // Data comes as dd-m-Y from controller, just display it
            return data;
          }
        },
        {
          // Class/Section
          targets: 7,
          responsivePriority: 8,
          render: function (data, type, full, meta) {
            return data || 'N/A';
          }
        },
        {
          // Fee Status - Badge
          targets: 8,
          responsivePriority: 9,
          className: 'text-center',
          orderable: false,
          render: function (data, type, full, meta) {
            return data;
          }
        },
        {
          // Pending Fee
          targets: 9,
          responsivePriority: 9,
          className: 'text-center',
          render: function (data, type, full, meta) {
            return data;
          }
        },
        {
          // Actions - History button
          targets: 10,
          title: 'Actions',
          orderable: false,
          searchable: false,
          responsivePriority: 11,
          className: 'text-center',
          render: function (data, type, full, meta) {
            var studentId = full['encrypted_id'] || full['id'];
            return (
              '<a href="javascript:void(0);" class="btn btn-sm btn-icon btn-text-info rounded-pill" onclick="viewFeeHistory(\'' +
              studentId +
              '\')" title="View Payment History">' +
              '<i class="ri-history-line ri-20px"></i>' +
              '</a>'
            );
          }
        }
      ],
      order: [[2, 'asc']], // Order by Roll No
      dom: '<"card-header flex-column flex-md-row border-bottom"<"head-label text-center"><"dt-action-buttons text-end pt-3 pt-md-0"B>><"row"<"col-sm-12 col-md-6 mt-5 mt-md-0"l><"col-sm-12 col-md-6 d-flex justify-content-center justify-content-md-end"f>>t<"row"<"col-sm-12 col-md-6"i><"col-sm-12 col-md-6"p>>',
      displayLength: 10,
      lengthMenu: [10, 25, 50, 75, 100],
      language: {
        paginate: {
          next: '<i class="ri-arrow-right-s-line"></i>',
          previous: '<i class="ri-arrow-left-s-line"></i>'
        },
        processing:
          '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div>'
      },
      buttons: [
        {
          extend: 'collection',
          className: 'btn btn-label-secondary dropdown-toggle ms-2 waves-effect waves-light',
          text: '<i class="ri-external-link-line me-sm-1"></i> <span class="d-none d-sm-inline-block">Export</span>',
          buttons: [
            {
              extend: 'print',
              text: '<i class="ri-printer-line me-1" ></i>Print',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              },
              customize: function (win) {
                $(win.document.body)
                  .css('color', config.colors.headingColor)
                  .css('border-color', config.colors.borderColor)
                  .css('background-color', config.colors.bodyBg);
                $(win.document.body)
                  .find('table')
                  .addClass('compact')
                  .css('color', 'inherit')
                  .css('border-color', 'inherit')
                  .css('background-color', 'inherit');
              }
            },
            {
              extend: 'csv',
              text: '<i class="ri-file-text-line me-1" ></i>Csv',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              }
            },
            {
              extend: 'excel',
              text: '<i class="ri-file-excel-line me-1"></i>Excel',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              }
            },
            {
              extend: 'pdf',
              text: '<i class="ri-file-pdf-line me-1"></i>Pdf',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              }
            },
            {
              extend: 'copy',
              text: '<i class="ri-file-copy-line me-1" ></i>Copy',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              }
            }
          ]
        }
      ],
      responsive: {
        details: {
          display: $.fn.dataTable.Responsive.display.modal({
            header: function (row) {
              var data = row.data();
              return 'Details of ' + data['name'];
            }
          }),
          type: 'column',
          renderer: function (api, rowIdx, columns) {
            var data = $.map(columns, function (col, i) {
              return col.title !== '' // ? Do not show row in modal popup if title is blank (for check box)
                ? '<tr data-dt-row="' +
                    col.rowIndex +
                    '" data-dt-column="' +
                    col.columnIndex +
                    '">' +
                    '<td>' +
                    col.title +
                    ':' +
                    '</td> ' +
                    '<td>' +
                    col.data +
                    '</td>' +
                    '</tr>'
                : '';
            }).join('');

            return data ? $('<table class="table"/><tbody />').append(data) : false;
          }
        }
      }
    });

    $('div.head-label').html('<h5 class="card-title mb-0">Student Fee Management</h5>');
  }

  // Delete Record
  $('.datatables-basic tbody').on('click', '.delete-record', function () {
    dt_basic.row($(this).parents('tr')).remove().draw();
  });
});

// Make viewFeeHistory function global
window.viewFeeHistory = viewFeeHistory;
