﻿/**
 * DataTables for Students List - Server Side Processing
 */

'use strict';

let fv, studentModal, dt_basic;

// Toast notification helper
function showStudentToast(message, type = 'success') {
  const toastElement = document.getElementById('studentToast');
  const toastTitle = document.getElementById('toastTitle');
  const toastIcon = document.getElementById('toastIcon');
  const toastMessage = document.getElementById('toastMessage');

  if (!toastElement) {
    console.error('Toast element not found');
    return;
  }

  // Set message
  toastMessage.textContent = message;

  // Set color and icon based on type
  if (type === 'success') {
    toastTitle.textContent = 'Success';
    toastIcon.className = 'ri-checkbox-circle-fill me-2 text-success';
    toastElement.classList.add('text-success');
  } else if (type === 'edit') {
    toastTitle.textContent = 'Success';
    toastIcon.className = 'ri-edit-2-fill me-2 text-info';
    toastElement.classList.add('text-info');
  } else if (type === 'error') {
    toastTitle.textContent = 'Error';
    toastIcon.className = 'ri-error-warning-fill me-2 text-danger';
    toastElement.classList.add('text-danger');
  }

  // Show toast
  const toast = new bootstrap.Toast(toastElement);
  toast.show();
}

document.addEventListener('DOMContentLoaded', function (e) {
  (function () {
    const formAddNewRecord = document.getElementById('form-add-new-record');
    const studentModalElement = document.getElementById('studentModal');

    setTimeout(() => {
      const newRecord = document.querySelector('.create-new');

      // Open modal to add new student
      if (newRecord) {
        newRecord.addEventListener('click', function () {
          // Load form via AJAX
          fetch('/admin/students/create', {
            headers: {
              'X-Requested-With': 'XMLHttpRequest',
              Accept: 'text/html'
            }
          })
            .then(response => response.text())
            .then(html => {
              document.getElementById('studentModalContent').innerHTML = html;

              // Initialize modal
              studentModal = new bootstrap.Modal(studentModalElement);

              // Add event listener to stop camera when modal is closed
              studentModalElement.addEventListener('hidden.bs.modal', function () {
                console.log('Modal hidden event triggered');
                if (typeof window.stopCameraStream === 'function') {
                  window.stopCameraStream();
                }
              });

              studentModal.show();

              // Re-initialize Select2 and other form elements
              initializeFormElements();
            })
            .catch(error => {
              console.error('Error loading form:', error);
              alert('Failed to load student form');
            });
        });
      }
    }, 200);
  })();
});

// Initialize form elements (Select2, FlatPickr, Camera, Chain Select, etc.)
function initializeFormElements() {
  // Initialize Select2 only for elements inside the modal
  if (typeof $.fn.select2 !== 'undefined') {
    var modalContent = document.getElementById('studentModalContent');
    if (modalContent) {
      var select2Elements = modalContent.querySelectorAll('.select2');
      select2Elements.forEach(function (element) {
        var $element = $(element);
        // Check if already initialized
        if (!$element.hasClass('select2-hidden-accessible')) {
          $element.select2({
            placeholder: $element.data('placeholder') || 'Select value',
            dropdownParent: $element.closest('.modal-body') || $element.parent()
          });
        }
      });
    }
  }

  // Initialize FlatPickr if present
  const flatpickrDate = document.querySelector('[name="basicDate"]');
  if (flatpickrDate && typeof flatpickr !== 'undefined') {
    flatpickr(flatpickrDate, {
      enableTime: false,
      dateFormat: 'm/d/Y'
    });
  }

  // Reinitialize chain select for dynamically loaded forms
  if (typeof window.ChainSelect !== 'undefined') {
    console.log('Reinitializing ChainSelect for dynamic form');
    // Re-initialize the chain select system for the newly loaded form
    window.chainSelectInstance = new window.ChainSelect();
  }

  // Reinitialize camera if initializeCamera function is available (from dynamic form)
  if (typeof window.initializeCamera === 'function') {
    window.initializeCamera();
  }

  // Attach submit handlers to forms directly
  var addForm = document.getElementById('form-add-new-record');
  if (addForm) {
    console.log('Attaching submit handler to add form');
    addForm.onsubmit = handleFormSubmit;
  }

  var editForm = document.getElementById('form-edit-record');
  if (editForm) {
    console.log('Attaching submit handler to edit form');
    editForm.onsubmit = handleFormSubmit;
  }
}

// Edit student - Load form in modal
function editStudent(encryptedId) {
  const studentModalElement = document.getElementById('studentModal');

  fetch(`/admin/student/edit/${encryptedId}`, {
    headers: {
      'X-Requested-With': 'XMLHttpRequest',
      Accept: 'text/html'
    }
  })
    .then(response => response.text())
    .then(html => {
      document.getElementById('studentModalContent').innerHTML = html;

      // Initialize modal
      studentModal = new bootstrap.Modal(studentModalElement);

      // Add event listener to stop camera when modal is closed
      studentModalElement.addEventListener('hidden.bs.modal', function () {
        console.log('Modal hidden event triggered');
        if (typeof window.stopCameraStream === 'function') {
          window.stopCameraStream();
        }
      });

      studentModal.show();

      // Re-initialize form elements
      initializeFormElements();
    })
    .catch(error => {
      console.error('Error loading edit form:', error);
      alert('Failed to load student edit form');
    });
}

// Global camera stream state
window.cameraStreamState = { stream: null };

// Open Camera function
window.openCameraEdit = async function (e) {
  e.preventDefault();
  e.stopPropagation();
  console.log('Open Camera clicked');
  try {
    window.cameraStreamState.stream = await navigator.mediaDevices.getUserMedia({
      video: { facingMode: 'user' },
      audio: false
    });
    const video = document.getElementById('video');
    if (video) {
      video.srcObject = window.cameraStreamState.stream;
    }
    const container = document.getElementById('cameraContainer');
    if (container) {
      container.style.display = 'block';
    }
    const captureBtn = document.getElementById('captureBtn');
    if (captureBtn) {
      captureBtn.classList.remove('d-none');
    }
    const closeBtn = document.getElementById('closeCamera');
    if (closeBtn) {
      closeBtn.classList.remove('d-none');
    }
    const openBtn = document.getElementById('openCamera');
    if (openBtn) {
      openBtn.classList.add('d-none');
    }
    console.log('Camera stream started');
  } catch (err) {
    console.error('Camera access error:', err);
    alert('Could not access camera: ' + err.message);
  }
};

// Capture function
window.captureCameraEdit = function (e) {
  e.preventDefault();
  e.stopPropagation();
  console.log('Capture clicked');
  try {
    const video = document.getElementById('video');
    const canvas = document.getElementById('canvas');
    const previewImage = document.getElementById('previewImage');
    const imageDataInput = document.getElementById('image_data');
    const fileInput = document.getElementById('image');

    if (!canvas || !video) {
      console.error('Canvas or video not found');
      return;
    }

    const context = canvas.getContext('2d');
    canvas.width = video.videoWidth || 280;
    canvas.height = video.videoHeight || 210;
    context.drawImage(video, 0, 0, canvas.width, canvas.height);
    const dataUrl = canvas.toDataURL('image/jpeg', 0.9);

    if (previewImage) {
      previewImage.src = dataUrl;
      previewImage.style.display = 'block';
    }
    if (imageDataInput) {
      imageDataInput.value = dataUrl;
    }
    if (fileInput) {
      fileInput.value = '';
    }
    console.log('Image captured');
  } catch (err) {
    console.error('Capture error:', err);
    alert('Failed to capture image: ' + err.message);
  }
};

// Close Camera function
window.closeCameraEdit = function (e) {
  e.preventDefault();
  e.stopPropagation();
  console.log('Close Camera clicked');
  if (window.cameraStreamState.stream) {
    window.cameraStreamState.stream.getTracks().forEach(track => track.stop());
    window.cameraStreamState.stream = null;
  }
  const container = document.getElementById('cameraContainer');
  if (container) {
    container.style.display = 'none';
  }
  const captureBtn = document.getElementById('captureBtn');
  if (captureBtn) {
    captureBtn.classList.add('d-none');
  }
  const closeBtn = document.getElementById('closeCamera');
  if (closeBtn) {
    closeBtn.classList.add('d-none');
  }
  const openBtn = document.getElementById('openCamera');
  if (openBtn) {
    openBtn.classList.remove('d-none');
  }
  console.log('Camera closed');
};

// Stop Camera Stream function (for modal cleanup)
window.stopCameraStream = function () {
  console.log('Stopping camera stream...');
  try {
    if (window.cameraStreamState && window.cameraStreamState.stream) {
      window.cameraStreamState.stream.getTracks().forEach(track => {
        console.log('Stopping track:', track.kind);
        track.stop();
      });
      window.cameraStreamState.stream = null;
    }
    // Reset camera UI
    const container = document.getElementById('cameraContainer');
    if (container) {
      container.style.display = 'none';
    }
    const captureBtn = document.getElementById('captureBtn');
    if (captureBtn) {
      captureBtn.classList.add('d-none');
    }
    const closeBtn = document.getElementById('closeCamera');
    if (closeBtn) {
      closeBtn.classList.add('d-none');
    }
    const openBtn = document.getElementById('openCamera');
    if (openBtn) {
      openBtn.classList.remove('d-none');
    }
    console.log('Camera stream stopped');
  } catch (err) {
    console.error('Error stopping camera stream:', err);
  }
};

// DataTable (jQuery)
$(function () {
  var dt_basic_table = $('.datatables-basic');

  // Handle filter button click
  $('#filterBtn').on('click', function () {
    var classId = $('#filterClass').val();
    var sectionId = $('#filterSection').val();

    console.log('Filter applied - Class:', classId, 'Section:', sectionId);

    // Reload DataTable with filters
    if (dt_basic) {
      dt_basic.ajax.reload(function () {
        console.log('DataTable reloaded with filters');
      });
    }
  });

  // Handle clear filters button click
  $('#clearFilterBtn').on('click', function () {
    console.log('Clearing all filters');

    // Reset all filter dropdowns
    $('#filterClass').val('').trigger('change.select2');
    $('#filterSection').val('').trigger('change.select2');

    // Reload DataTable without filters
    if (dt_basic) {
      dt_basic.ajax.reload(function () {
        console.log('DataTable reloaded - showing all records');
      });
    }
  });

  // DataTable with server-side processing
  if (dt_basic_table.length) {
    dt_basic = dt_basic_table.DataTable({
      processing: true,
      serverSide: true,
      order: [[2, 'desc']], // Sort by Roll No (column 2) in descending order
      ajax: {
        url: baseUrl + 'admin/students/data',
        type: 'GET',
        data: function (d) {
          // Add filter parameters
          d.class_id = $('#filterClass').val();
          d.section_id = $('#filterSection').val();
          console.log('DataTable request:', d); // Log the request data
          return d;
        }
      },
      columns: [
        { data: '' }, // Responsive control
        { data: 'id' }, // Checkbox
        { data: 'roll_no' }, // Roll No
        { data: 'name' }, // Name with image
        { data: 'fname' }, // Father Name
        { data: 'mobile' }, // Mobile
        { data: 'join_date' }, // Join Date
        { data: 'class_section' }, // Class/Section
        { data: 'fstatus' }, // Status
        { data: '' }, // Profile
        { data: '' } // Actions
      ],
      columnDefs: [
        {
          // For Responsive
          className: 'control',
          orderable: false,
          searchable: false,
          responsivePriority: 2,
          targets: 0,
          render: function (data, type, full, meta) {
            return '';
          }
        },
        {
          // For Checkboxes
          targets: 1,
          orderable: false,
          searchable: false,
          responsivePriority: 3,
          checkboxes: true,
          render: function () {
            return '<input type="checkbox" class="dt-checkboxes form-check-input">';
          },
          checkboxes: {
            selectAllRender: '<input type="checkbox" class="form-check-input">'
          }
        },
        {
          // Roll No
          targets: 2,
          orderable: true,
          responsivePriority: 4,
          className: 'text-center',
          render: function (data, type, full, meta) {
            return '<span class="fw-medium">' + (data || 'N/A') + '</span>';
          }
        },
        {
          // Name with Avatar image
          targets: 3,
          responsivePriority: 1,
          render: function (data, type, full, meta) {
            var $user_img = full['image'];
            var $name = full['name'];

            var $output = '<img src="' + $user_img + '" alt="Avatar" class="rounded-circle" width="32" height="32">';

            // Creates full output for row
            var $row_output =
              '<div class="d-flex justify-content-start align-items-center user-name">' +
              '<div class="avatar-wrapper">' +
              '<div class="avatar me-2">' +
              $output +
              '</div>' +
              '</div>' +
              '<div class="d-flex flex-column">' +
              '<span class="emp_name text-truncate text-heading fw-medium">' +
              $name +
              '</span>' +
              '</div>' +
              '</div>';
            return $row_output;
          }
        },
        {
          // Father Name
          targets: 4,
          responsivePriority: 5,
          render: function (data, type, full, meta) {
            return data || 'N/A';
          }
        },
        {
          // Mobile Numbers
          targets: 5,
          responsivePriority: 6,
          width: '120px',
          className: 'text-center',
          render: function (data, type, full, meta) {
            return data || 'N/A';
          }
        },
        {
          // Join Date
          targets: 6,
          responsivePriority: 7,
          className: 'text-center',
          render: function (data, type, full, meta) {
            if (!data) return 'N/A';
            // Data comes as dd-m-Y from controller, just display it
            return data;
          }
        },
        {
          // Class/Section
          targets: 7,
          responsivePriority: 8,
          render: function (data, type, full, meta) {
            return data || 'N/A';
          }
        },
        {
          // Fee Status - Badge
          targets: 8,
          responsivePriority: 9,
          className: 'text-center',
          orderable: false,
          render: function (data, type, full, meta) {
            return data;
          }
        },
        {
          // Profile View Icon
          targets: 9,
          title: 'Profile',
          orderable: false,
          searchable: false,
          responsivePriority: 10,
          className: 'text-center',
          render: function (data, type, full, meta) {
            var encryptedId = full['encrypted_id'];
            return (
              '<a href="/admin/student/profile/' +
              encryptedId +
              '" target="_blank" class="btn btn-sm btn-icon btn-text-secondary rounded-pill" title="View Profile">' +
              '<i class="ri-eye-line ri-20px"></i>' +
              '</a>'
            );
          }
        },
        {
          // Actions - Edit button
          targets: 10,
          title: 'Actions',
          orderable: false,
          searchable: false,
          responsivePriority: 11,
          className: 'text-center',
          render: function (data, type, full, meta) {
            var encryptedId = full['encrypted_id'];
            return (
              '<a href="javascript:void(0);" class="btn btn-sm btn-icon btn-text-secondary rounded-pill item-edit" onclick="editStudent(\'' +
              encryptedId +
              '\')" title="Edit Student">' +
              '<i class="ri-edit-box-line ri-20px"></i>' +
              '</a>'
            );
          }
        }
      ],
      order: [[2, 'asc']], // Order by Roll No
      dom: '<"card-header flex-column flex-md-row border-bottom"<"head-label text-center"><"dt-action-buttons text-end pt-3 pt-md-0"B>><"row"<"col-sm-12 col-md-6 mt-5 mt-md-0"l><"col-sm-12 col-md-6 d-flex justify-content-center justify-content-md-end"f>>t<"row"<"col-sm-12 col-md-6"i><"col-sm-12 col-md-6"p>>',
      displayLength: 10,
      lengthMenu: [10, 25, 50, 75, 100],
      language: {
        paginate: {
          next: '<i class="ri-arrow-right-s-line"></i>',
          previous: '<i class="ri-arrow-left-s-line"></i>'
        },
        processing:
          '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div>'
      },
      buttons: [
        {
          text: '<i class="ri-add-line ri-16px me-sm-2"></i> <span class="d-none d-sm-inline-block">Add New Student</span>',
          className: 'create-new btn btn-primary waves-effect waves-light'
        },
        {
          extend: 'collection',
          className: 'btn btn-label-secondary dropdown-toggle ms-2 waves-effect waves-light',
          text: '<i class="ri-external-link-line me-sm-1"></i> <span class="d-none d-sm-inline-block">Export</span>',
          buttons: [
            {
              extend: 'print',
              text: '<i class="ri-printer-line me-1" ></i>Print',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              },
              customize: function (win) {
                $(win.document.body)
                  .css('color', config.colors.headingColor)
                  .css('border-color', config.colors.borderColor)
                  .css('background-color', config.colors.bodyBg);
                $(win.document.body)
                  .find('table')
                  .addClass('compact')
                  .css('color', 'inherit')
                  .css('border-color', 'inherit')
                  .css('background-color', 'inherit');
              }
            },
            {
              extend: 'csv',
              text: '<i class="ri-file-text-line me-1" ></i>Csv',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              }
            },
            {
              extend: 'excel',
              text: '<i class="ri-file-excel-line me-1"></i>Excel',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              }
            },
            {
              extend: 'pdf',
              text: '<i class="ri-file-pdf-line me-1"></i>Pdf',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              }
            },
            {
              extend: 'copy',
              text: '<i class="ri-file-copy-line me-1" ></i>Copy',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7],
                format: {
                  body: function (inner, coldex, rowdex) {
                    if (inner.length <= 0) return inner;
                    var el = $.parseHTML(inner);
                    var result = '';
                    $.each(el, function (index, item) {
                      if (item.classList !== undefined && item.classList.contains('user-name')) {
                        result = result + item.lastChild.firstChild.textContent;
                      } else if (item.innerText === undefined) {
                        result = result + item.textContent;
                      } else result = result + item.innerText;
                    });
                    return result;
                  }
                }
              }
            }
          ]
        }
      ],
      responsive: {
        details: {
          display: $.fn.dataTable.Responsive.display.modal({
            header: function (row) {
              var data = row.data();
              return 'Details of ' + data['name'];
            }
          }),
          type: 'column',
          renderer: function (api, rowIdx, columns) {
            var data = $.map(columns, function (col, i) {
              return col.title !== '' // ? Do not show row in modal popup if title is blank (for check box)
                ? '<tr data-dt-row="' +
                    col.rowIndex +
                    '" data-dt-column="' +
                    col.columnIndex +
                    '">' +
                    '<td>' +
                    col.title +
                    ':' +
                    '</td> ' +
                    '<td>' +
                    col.data +
                    '</td>' +
                    '</tr>'
                : '';
            }).join('');

            return data ? $('<table class="table"/><tbody />').append(data) : false;
          }
        }
      }
    });

    $('div.head-label').html('<h5 class="card-title mb-0">Active Students List</h5>');
  }

  // Delete Record
  $('.datatables-basic tbody').on('click', '.delete-record', function () {
    dt_basic.row($(this).parents('tr')).remove().draw();
  });
});

// Form submission handler for both add and edit
function handleFormSubmit(e) {
  e.preventDefault();
  console.log('FORM SUBMIT TRIGGERED', { formId: this.id });

  var $form = $(this);
  var url = $form.attr('action');
  var method = $form.find('input[name="_method"]').val() || 'POST';

  console.log('Submitting to:', url, 'with method:', method);

  // Check if there's a file input with a file selected
  var fileInput = $form.find('input[type="file"]')[0];
  var hasFile = fileInput?.files?.length > 0;

  console.log('=== DEBUGGING INPUT VALUES ===');
  var allInputs = this.querySelectorAll('input[name], textarea[name], select[name]');
  allInputs.forEach(function (input) {
    console.log('Input:', {
      name: input.name,
      type: input.type,
      value: input.value,
      disabled: input.disabled,
      visible: input.offsetParent !== null
    });
  });

  if (hasFile) {
    // For file uploads, create FormData and populate it manually
    var formData = new FormData();

    // Add all text inputs, selects, and textareas
    $form.find('input[type!=file], textarea, select').each(function () {
      if (this.name && !this.disabled) {
        if (this.type === 'checkbox' || this.type === 'radio') {
          if (this.checked) {
            formData.append(this.name, this.value);
          }
        } else {
          formData.append(this.name, this.value);
        }
      }
    });

    // Add the file
    if (fileInput.files[0]) {
      formData.append('image', fileInput.files[0]);
    }

    // Add image_data if present
    var imageDataInput = $form.find('input[name="image_data"]');
    if (imageDataInput.length) {
      formData.append('image_data', imageDataInput.val());
    }

    console.log('=== FORMDATA BEING SENT (with file) ===');
    for (var pair of formData.entries()) {
      if (pair[1] instanceof File) {
        console.log('  ' + pair[0] + ': File(' + pair[1].name + ')');
      } else {
        console.log('  ' + pair[0] + ': "' + pair[1] + '"');
      }
    }

    $.ajax({
      url: url,
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      headers: {
        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
      },
      success: function (response) {
        console.log('Form submission success:', response);

        // Close camera stream if open
        if (window.cameraStreamState && window.cameraStreamState.stream) {
          window.cameraStreamState.stream.getTracks().forEach(track => track.stop());
          window.cameraStreamState.stream = null;
          console.log('Camera stream closed');
        }

        // Show success toast
        showStudentToast('Student added successfully!', 'success');

        // Find and close the modal by looking for studentModal
        var studentModalElement = document.getElementById('studentModal');
        if (studentModalElement) {
          var modalInstance = bootstrap.Modal.getInstance(studentModalElement);
          if (modalInstance) {
            modalInstance.hide();
          }
        }

        // Update specific row without full table reload
        if (dt_basic && response.student_id && response.row_data) {
          var rowId = 'row_' + response.student_id;
          var rowNode = document.getElementById(rowId);
          if (rowNode) {
            console.log('Updating specific row:', rowId);
            // Update the row with new data without reloading table
            var row = dt_basic.row(rowNode);
            row.data(response.row_data).draw(false);
            console.log('Row updated successfully, staying on current page');
          } else {
            // If row not found (new record), reload entire table
            console.log('Row not found, reloading entire table');
            var currentPage = dt_basic.page();
            dt_basic.ajax.reload(function () {
              dt_basic.page(currentPage).draw(false);
            });
          }
        } else {
          // Fallback: reload entire table if no row_data in response
          console.log('No row_data in response, reloading entire table');
          var currentPage = dt_basic.page();
          dt_basic.ajax.reload(function () {
            dt_basic.page(currentPage).draw(false);
          });
        }
      },
      error: function (xhr) {
        console.error('Form submission error:', xhr);
        console.log('=== SERVER ERROR RESPONSE ===');
        console.log('Status:', xhr.status);
        console.log('Response:', xhr.responseJSON);
        console.log('Response Text:', xhr.responseText);

        var errorMessage = 'Failed to save student.';

        if (xhr.status === 422 && xhr.responseJSON && xhr.responseJSON.errors) {
          var errors = xhr.responseJSON.errors;
          console.log('Validation errors:', errors);
          var errorLines = [];
          for (var field in errors) {
            if (Array.isArray(errors[field])) {
              errorLines.push(errors[field].join(', '));
            } else {
              errorLines.push(errors[field]);
            }
          }
          errorMessage = errorLines.join('\n');
        } else if (xhr.responseJSON && xhr.responseJSON.error) {
          errorMessage = xhr.responseJSON.error;
        } else if (xhr.responseText) {
          errorMessage = xhr.responseText.substring(0, 300);
        }

        alert('Error: ' + errorMessage);
      }
    });
  } else {
    // No file - use jQuery serialize
    var ajaxData = $form.serialize();
    console.log('=== SERIALIZED DATA BEING SENT (no file) ===');
    console.log(ajaxData);

    $.ajax({
      url: url,
      type: method,
      data: ajaxData,
      processData: true,
      contentType: 'application/x-www-form-urlencoded; charset=UTF-8',
      headers: {
        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
      },
      success: function (response) {
        console.log('Form submission success:', response);

        // Close camera stream if open
        if (window.cameraStreamState && window.cameraStreamState.stream) {
          window.cameraStreamState.stream.getTracks().forEach(track => track.stop());
          window.cameraStreamState.stream = null;
          console.log('Camera stream closed');
        }

        // Show success toast for edit
        showStudentToast('Student updated successfully!', 'edit');

        // Find and close the modal by looking for studentModal
        var studentModalElement = document.getElementById('studentModal');
        if (studentModalElement) {
          var modalInstance = bootstrap.Modal.getInstance(studentModalElement);
          if (modalInstance) {
            modalInstance.hide();
          }
        }

        // Update specific row without full table reload
        if (dt_basic && response.student_id && response.row_data) {
          var rowId = 'row_' + response.student_id;
          var rowNode = document.getElementById(rowId);
          if (rowNode) {
            console.log('Updating specific row:', rowId);
            // Update the row with new data without reloading table
            var row = dt_basic.row(rowNode);
            row.data(response.row_data).draw(false);
            console.log('Row updated successfully, staying on current page');
          } else {
            // If row not found (new record), reload entire table
            console.log('Row not found, reloading entire table');
            var currentPage = dt_basic.page();
            dt_basic.ajax.reload(function () {
              dt_basic.page(currentPage).draw(false);
            });
          }
        } else {
          // Fallback: reload entire table if no row_data in response
          console.log('No row_data in response, reloading entire table');
          var currentPage = dt_basic.page();
          dt_basic.ajax.reload(function () {
            dt_basic.page(currentPage).draw(false);
          });
        }
      },
      error: function (xhr) {
        console.error('Form submission error:', xhr);
        console.log('=== SERVER ERROR RESPONSE ===');
        console.log('Status:', xhr.status);
        console.log('Response:', xhr.responseJSON);
        console.log('Response Text:', xhr.responseText);

        var errorMessage = 'Failed to save student.';

        if (xhr.status === 422 && xhr.responseJSON && xhr.responseJSON.errors) {
          var errors = xhr.responseJSON.errors;
          console.log('Validation errors:', errors);
          var errorLines = [];
          for (var field in errors) {
            if (Array.isArray(errors[field])) {
              errorLines.push(errors[field].join(', '));
            } else {
              errorLines.push(errors[field]);
            }
          }
          errorMessage = errorLines.join('\n');
        } else if (xhr.responseJSON && xhr.responseJSON.error) {
          errorMessage = xhr.responseJSON.error;
        } else if (xhr.responseText) {
          errorMessage = xhr.responseText.substring(0, 300);
        }

        alert('Error: ' + errorMessage);
      }
    });
  }
}

// Make editStudent function global
window.editStudent = editStudent;
