/**
 * DataTables for Fee Challans - Server Side Processing
 */

'use strict';

let dt_basic;

// Function to refresh statistics
function refreshChallanStats() {
  fetch('/admin/student/challans/stats', {
    method: 'GET',
    headers: {
      'X-Requested-With': 'XMLHttpRequest',
      Accept: 'application/json'
    }
  })
    .then(response => response.json())
    .then(stats => {
      // Update statistics cards
      document.getElementById('stat-total-challans').textContent = stats.total_challans || 0;
      document.getElementById('stat-active-challans').textContent = stats.issued + stats.pending || 0;
      document.getElementById('stat-active-details').textContent =
        `Issued: ${stats.issued || 0} | Pending: ${stats.pending || 0}`;
      document.getElementById('stat-paid-amount').textContent =
        '₨' + (stats.paid_amount ? Number(stats.paid_amount).toLocaleString() : '0');
      document.getElementById('stat-paid-details').textContent = `${stats.paid || 0} paid challans`;
      document.getElementById('stat-unpaid-amount').textContent =
        '₨' + (stats.unpaid_amount ? Number(stats.unpaid_amount).toLocaleString() : '0');

      console.log('Statistics refreshed successfully');
    })
    .catch(error => {
      console.error('Error refreshing stats:', error);
    });
}

// Global function to reload challans table
window.reloadChallansTable = function () {
  console.log('reloadChallansTable called, dt_basic:', typeof dt_basic);
  if (typeof dt_basic !== 'undefined' && dt_basic) {
    console.log('Reloading DataTable via global function...');
    dt_basic.ajax.reload(null, true);
    // Refresh statistics after table reload
    refreshChallanStats();
    return true;
  }
  console.error('dt_basic is not available');
  return false;
};

// View payment history in modal
window.viewPaymentHistory = function (studentId) {
  const modalElement = document.getElementById('paymentHistoryModal');

  fetch(`/admin/student/payment-history/${studentId}`, {
    headers: {
      'X-Requested-With': 'XMLHttpRequest',
      Accept: 'text/html'
    }
  })
    .then(response => response.text())
    .then(html => {
      document.getElementById('paymentHistoryModalContent').innerHTML = html;
      let modal = new bootstrap.Modal(modalElement);
      modal.show();
    })
    .catch(error => {
      console.error('Error loading payment history:', error);
      showChallanToast('Failed to load payment history', 'error');
    });
};

// Toast notification helper
function showChallanToast(message, type = 'success') {
  const toastElement = document.getElementById('challanToast');
  const toastTitle = document.getElementById('toastTitle');
  const toastIcon = document.getElementById('toastIcon');
  const toastMessage = document.getElementById('toastMessage');

  if (!toastElement) {
    console.error('Toast element not found');
    return;
  }

  toastMessage.textContent = message;

  if (type === 'success') {
    toastTitle.textContent = 'Success';
    toastIcon.className = 'ri-checkbox-circle-fill me-2 text-success';
  } else if (type === 'error') {
    toastTitle.textContent = 'Error';
    toastIcon.className = 'ri-error-warning-fill me-2 text-danger';
  }

  const toast = new bootstrap.Toast(toastElement);
  toast.show();
}

// Open challan form modal (create new)
window.openChallanForm = function () {
  loadChallanForm('/admin/student/challan/create');
};

// Edit challan (load existing)
window.editChallan = function (id) {
  loadChallanForm(`/admin/student/challan/edit/${id}`);
};

// Common function to load challan form
function loadChallanForm(url) {
  const challanModalElement = document.getElementById('challanModal');

  fetch(url, {
    headers: {
      'X-Requested-With': 'XMLHttpRequest',
      Accept: 'text/html'
    }
  })
    .then(response => {
      // Check if response is an error (403, 404, etc.)
      if (!response.ok) {
        return response.json().then(errorData => {
          throw errorData;
        });
      }
      return response.text();
    })
    .then(html => {
      $('#challanModalContent').html(html);
      let challanModal = new bootstrap.Modal(challanModalElement);
      challanModal.show();

      // Initialize Select2 in modal after a short delay to ensure DOM is ready
      setTimeout(() => {
        if (typeof $.fn.select2 !== 'undefined') {
          const modal = $('#challanModal');
          const modalContent = $('#challanModalContent');

          modalContent.find('.select2').each(function () {
            const $element = $(this);
            if (!$element.hasClass('select2-hidden-accessible')) {
              const placeholder =
                $element.attr('id') === 'app_student_id'
                  ? 'Select Student'
                  : $element.attr('id') === 'app_class_id'
                    ? 'Select Class'
                    : $element.attr('id') === 'app_section_id'
                      ? 'Select Section'
                      : 'Select option';

              $element.select2({
                placeholder: placeholder,
                dropdownParent: modal,
                width: '100%'
              });

              console.log(
                'Select2 initialized for',
                $element.attr('id'),
                'with',
                $element.find('option').length,
                'options'
              );
            }
          });
        }
      }, 100);
    })
    .catch(error => {
      console.error('Error loading form:', error);

      // If it's a permission error with a custom message
      if (error && error.error) {
        // Check if Swal (SweetAlert2) is available
        if (typeof Swal !== 'undefined') {
          Swal.fire({
            icon: 'error',
            title: 'Access Denied',
            text: error.error,
            confirmButtonText: 'OK',
            customClass: {
              confirmButton: 'btn btn-primary'
            }
          });
        } else {
          // Fallback to alert if Swal is not available
          alert('Access Denied\n\n' + error.error);
        }
      } else {
        showChallanToast('Failed to load challan form', 'error');
      }
    });
}

// Print challan
window.printChallan = function (id) {
  window.open(`/admin/student/challan/print/${id}`, '_blank');
};

// Delete challan
window.deleteChallan = function (id) {
  if (confirm('Are you sure you want to delete this challan?')) {
    fetch(`/admin/student/challan/${id}`, {
      method: 'DELETE',
      headers: {
        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
        'X-Requested-With': 'XMLHttpRequest'
      }
    })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          showChallanToast('Challan deleted successfully', 'success');
          if (dt_basic) {
            dt_basic.ajax.reload();
          }
        } else {
          showChallanToast(data.error || 'Failed to delete challan', 'error');
        }
      })
      .catch(error => {
        console.error('Error:', error);
        showChallanToast('An error occurred', 'error');
      });
  }
};

document.addEventListener('DOMContentLoaded', function () {
  // Listeners are handled via DataTables actions or global functions
});

// DataTable (jQuery)
$(function () {
  var dt_basic_table = $('.datatables-basic');

  // Handle filter button click
  $('#filterBtn').on('click', function () {
    var classId = $('#filterClass').val();
    var sectionId = $('#filterSection').val();
    var status = $('#filterStatus').val();

    if (dt_basic) {
      dt_basic.ajax.reload();
      refreshChallanStats();
    }
  });

  // Handle clear filters button click
  $('#clearFilterBtn').on('click', function () {
    $('#filterClass').val('').trigger('change.select2');
    $('#filterSection').val('').trigger('change.select2');
    $('#filterStatus').val('').trigger('change.select2');

    if (dt_basic) {
      dt_basic.ajax.reload();
      refreshChallanStats();
    }
  });

  // DataTable with server-side processing
  if (dt_basic_table.length) {
    dt_basic = dt_basic_table.DataTable({
      processing: true,
      serverSide: true,
      order: [[2, 'desc']], // Sort by Challan No
      ajax: {
        url: '/admin/student/challans/data',
        type: 'GET',
        data: function (d) {
          d.class_id = $('#filterClass').val();
          d.section_id = $('#filterSection').val();
          d.status = $('#filterStatus').val();
          return d;
        },
        error: function (xhr, status, error) {
          console.error('[ERROR] DataTable AJAX error:', status, error, xhr);
        }
      },
      columns: [
        { data: '' }, // Responsive control
        { data: 'id' }, // Checkbox
        { data: 'challan_no' }, // Challan No with Date
        { data: 'feemonth' }, // Month
        { data: 'amount' }, // Amount
        { data: 'student_name' }, // Student Name
        { data: 'class_section' }, // Class/Section
        { data: 'transid' }, // Transaction ID
        { data: 'status' }, // Status
        { data: 'activity' }, // Activity
        { data: '' } // Actions
      ],
      columnDefs: [
        {
          // Responsive
          className: 'control',
          orderable: false,
          searchable: false,
          responsivePriority: 2,
          targets: 0,
          render: function () {
            return '';
          }
        },
        {
          // Checkboxes
          targets: 1,
          orderable: false,
          searchable: false,
          responsivePriority: 3,
          checkboxes: true,
          render: function () {
            return '<input type="checkbox" class="dt-checkboxes form-check-input">';
          },
          checkboxes: {
            selectAllRender: '<input type="checkbox" class="form-check-input">'
          }
        },
        {
          // Challan No with Date
          targets: 2,
          responsivePriority: 1,
          className: 'text-center',
          width: '110px',
          render: function (data, type, row) {
            return (
              '<span class="fw-bold text-primary" style="font-size:0.85rem; white-space:nowrap; display:block;">' +
              (data || 'N/A') +
              '</span>' +
              '<small class="text-muted" style="font-size:0.75rem;">' +
              (row.created_date || '') +
              '</small>'
            );
          }
        },
        {
          // Month
          targets: 3,
          responsivePriority: 4,
          width: '80px',
          className: 'text-center',
          render: function (data) {
            if (!data) {
              return 'N/A';
            }

            var parts = data.split(' ');
            var month = parts[0] || data;
            var year = parts[1] || '';

            return (
              '<span class="fw-medium">' +
              month +
              '</span>' +
              (year ? '<br><small class="text-muted">' + year + '</small>' : '')
            );
          }
        },
        {
          // Amount
          targets: 4,
          responsivePriority: 5,
          className: 'text-start',
          width: '100px',
          render: function (data, type, row) {
            var amountHtml =
              '<strong style="font-size:0.9rem;">Rs. ' + (data ? parseFloat(data).toFixed(2) : '0.00') + '</strong>';

            // Add description if exists
            if (row.description) {
              amountHtml +=
                '<br><small class="text-muted" style="font-size:0.75rem;" title="' +
                row.description +
                '"><i class="ri-information-line"></i> ' +
                row.description.substring(0, 30) +
                (row.description.length > 30 ? '...' : '') +
                '</small>';
            }

            return amountHtml;
          }
        },
        {
          // Student Name with Avatar
          targets: 5,
          responsivePriority: 6,
          render: function (data, type, row) {
            var $user_img = row.student_image || '/assets/img/avatars/default-avatar.png';
            var $name = row.student_name || 'N/A';
            var $roll = row.student_roll || 'N/A';

            var $output = '<img src="' + $user_img + '" alt="Avatar" class="rounded-circle" width="32" height="32">';

            var $row_output =
              '<div class="d-flex justify-content-start align-items-center user-name">' +
              '<div class="avatar-wrapper">' +
              '<div class="avatar me-2">' +
              $output +
              '</div>' +
              '</div>' +
              '<div class="d-flex flex-column">' +
              '<span class="emp_name text-truncate text-heading fw-medium">' +
              $name +
              '</span>' +
              '<small class="text-muted">Roll: ' +
              $roll +
              '</small>' +
              '</div>' +
              '</div>';
            return $row_output;
          }
        },
        {
          // Class/Section
          targets: 6,
          responsivePriority: 7,
          render: function (data) {
            return data || 'N/A';
          }
        },
        {
          // Transaction ID
          targets: 7,
          responsivePriority: 8,
          width: '110px',
          render: function (data) {
            return '<span style="font-size:0.8rem; white-space:nowrap;">' + (data || '-') + '</span>';
          }
        },
        {
          // Status Badge
          targets: 8,
          responsivePriority: 9,
          className: 'text-center',
          orderable: false,
          render: function (data) {
            return data;
          }
        },
        {
          // Activity
          targets: 9,
          responsivePriority: 10,
          width: '100px',
          className: 'text-center',
          orderable: false,
          render: function (data) {
            return data || '<small class="text-muted">No activity</small>';
          }
        },
        {
          // Actions
          targets: 10,
          title: 'Actions',
          orderable: false,
          searchable: false,
          responsivePriority: 11,
          className: 'text-center',
          render: function (data, type, row) {
            const encryptedId = row.encrypted_id || row.id;
            const studentEncryptedId = row.student_encrypted_id || row.student_id;

            return `
              <div class="btn-group" role="group">
                <a href="javascript:void(0);" class="btn btn-sm btn-icon btn-text-primary rounded-pill" 
                   onclick="editChallan('${encryptedId}')" title="Edit Challan">
                  <i class="ri-edit-line ri-20px"></i>
                </a>
                <a href="javascript:void(0);" class="btn btn-sm btn-icon btn-text-info rounded-pill" 
                   onclick="viewPaymentHistory('${studentEncryptedId}')" title="View Payment History">
                  <i class="ri-history-line ri-20px"></i>
                </a>
                <a href="javascript:void(0);" class="btn btn-sm btn-icon btn-text-secondary rounded-pill" 
                   onclick="printChallan('${encryptedId}')" title="Print Challan">
                  <i class="ri-printer-line ri-20px"></i>
                </a>
              </div>
            `;
          }
        }
      ],
      dom: '<"card-header flex-column flex-md-row border-bottom"<"head-label text-center"><"dt-action-buttons text-end pt-3 pt-md-0"B>><"row"<"col-sm-12 col-md-6 mt-5 mt-md-0"l><"col-sm-12 col-md-6 d-flex justify-content-center justify-content-md-end"f>>t<"row"<"col-sm-12 col-md-6"i><"col-sm-12 col-md-6"p>>',
      displayLength: 10,
      lengthMenu: [10, 25, 50, 75, 100],
      language: {
        paginate: {
          next: '<i class="ri-arrow-right-s-line"></i>',
          previous: '<i class="ri-arrow-left-s-line"></i>'
        },
        processing:
          '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div>'
      },
      buttons: [
        {
          text: '<i class="ri-add-line ri-16px me-sm-2"></i> <span class="d-none d-sm-inline-block">Create Challan</span>',
          className: 'create-new btn btn-primary waves-effect waves-light',
          action: function (e, dt, node, config) {
            openChallanForm();
          }
        },
        {
          extend: 'collection',
          className: 'btn btn-label-secondary dropdown-toggle ms-2 waves-effect waves-light',
          text: '<i class="ri-external-link-line me-sm-1"></i> <span class="d-none d-sm-inline-block">Export</span>',
          buttons: [
            {
              extend: 'print',
              text: '<i class="ri-printer-line me-1" ></i>Print',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7, 8, 9]
              }
            },
            {
              extend: 'csv',
              text: '<i class="ri-file-text-line me-1" ></i>Csv',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7, 8, 9]
              }
            },
            {
              extend: 'excel',
              text: '<i class="ri-file-excel-line me-1"></i>Excel',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7, 8, 9]
              }
            },
            {
              extend: 'pdf',
              text: '<i class="ri-file-pdf-line me-1"></i>Pdf',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7, 8, 9]
              }
            },
            {
              extend: 'copy',
              text: '<i class="ri-file-copy-line me-1" ></i>Copy',
              className: 'dropdown-item',
              exportOptions: {
                columns: [2, 3, 4, 5, 6, 7, 8, 9]
              }
            }
          ]
        }
      ],
      responsive: {
        details: {
          display: $.fn.dataTable.Responsive.display.modal({
            header: function (row) {
              var data = row.data();
              return 'Challan ' + data.challan_no;
            }
          }),
          type: 'column',
          renderer: function (api, rowIdx, columns) {
            var data = $.map(columns, function (col) {
              return col.title !== ''
                ? '<tr data-dt-row="' +
                    col.rowIndex +
                    '" data-dt-column="' +
                    col.columnIndex +
                    '">' +
                    '<td>' +
                    col.title +
                    ':' +
                    '</td> ' +
                    '<td>' +
                    col.data +
                    '</td>' +
                    '</tr>'
                : '';
            }).join('');

            return data ? $('<table class="table"/><tbody />').append(data) : false;
          }
        }
      }
    });

    $('div.head-label').html('<h5 class="card-title mb-0">Fee Challans List</h5>');
  }
});

// Make functions global
window.openChallanForm = openChallanForm;
window.printChallan = printChallan;
window.deleteChallan = deleteChallan;
