/**
 * Chain Select - Dynamic dependent dropdown handler
 * Handles unlimited cascading select elements
 */

(function () {
  'use strict';

  class ChainSelect {
    constructor() {
      this.selects = new Map();
      this.isLoadingInitial = false;
      this.init();
    }

    init() {
      // Find all chain select elements (exclude AJAX-enabled selects)
      const chainSelects = document.querySelectorAll('select[data-chain-select="true"]:not([data-ajax-search="true"])');

      console.log('[ChainSelect] Found', chainSelects.length, 'chain select elements (excluding AJAX)');

      chainSelects.forEach(select => {
        const name = select.name.replace('[]', '');
        const dependsOn = select.dataset.dependsOn;
        let initialValue = select.dataset.initialValue;

        // Parse JSON if it's an array
        if (initialValue && initialValue.startsWith('[')) {
          try {
            initialValue = JSON.parse(initialValue);
          } catch (e) {
            console.warn('[ChainSelect] Failed to parse initial value as JSON:', initialValue);
          }
        }

        console.log(
          '[ChainSelect] Initializing:',
          name,
          'depends on:',
          dependsOn || 'none',
          'initial:',
          initialValue || 'none'
        );

        this.selects.set(name, {
          element: select,
          dependsOn: dependsOn,
          model: select.dataset.model,
          filterColumn: select.dataset.filterColumn,
          display: select.dataset.display,
          key: select.dataset.key,
          initialValue: initialValue
        });

        // Disable dependent selects initially
        select.disabled = true;

        // Set up parent listener
        if (dependsOn) {
          this.setupParentListener(name, dependsOn);
        }
      });

      // Enable independent selects (no dependsOn) that are not AJAX
      document
        .querySelectorAll('select[data-chain-select="true"]:not([data-depends-on]):not([data-ajax-search="true"])')
        .forEach(select => {
          select.disabled = false;
          console.log('[ChainSelect] Enabled independent select:', select.name);
        });

      console.log('[ChainSelect] Initialization complete');

      // Load initial values for edit mode
      this.loadInitialValues();
    }
    setupParentListener(childName, parentName) {
      // Find parent select by name
      const parentSelect = document.querySelector(`select[name="${parentName}"], select[name="${parentName}[]"]`);

      if (!parentSelect) {
        console.warn(`Chain Select: Parent select "${parentName}" not found for "${childName}"`);
        return;
      }

      // Use jQuery events for select2 compatibility
      const $parentSelect = $(parentSelect);

      // Handler function for both native and select2 events
      const handleChange = () => {
        // Handle both single and multiple select
        const parentValue = parentSelect.multiple
          ? Array.from(parentSelect.selectedOptions)
              .map(opt => opt.value)
              .filter(v => v !== '')
          : parentSelect.value;

        this.handleParentChange(childName, parentValue);
      };

      // Listen to both native change and select2 events
      $parentSelect.on('change select2:select select2:unselect select2:clear', handleChange);
    }

    handleParentChange(childName, parentValue) {
      const childData = this.selects.get(childName);
      if (!childData) return;

      const childSelect = childData.element;

      // Don't clear if we're loading initial values
      if (this.isLoadingInitial) {
        console.log('[ChainSelect] Skipping clear during initial load for', childName);
        return;
      }

      // Clear child select
      this.clearSelect(childSelect);

      // Find all children that depend on this child and reset them
      this.resetDependentChildren(childName);

      // Check if parent has values (handle both array and single value)
      const hasValue = Array.isArray(parentValue) ? parentValue.length > 0 : !!parentValue;

      if (!hasValue) {
        childSelect.disabled = true;
        return;
      }

      // Load options for child
      this.loadOptions(childName, parentValue);
    }

    resetDependentChildren(parentName) {
      this.selects.forEach((data, name) => {
        if (data.dependsOn === parentName) {
          this.clearSelect(data.element);
          data.element.disabled = true;
          // Recursively reset children of this child
          this.resetDependentChildren(name);
        }
      });
    }

    clearSelect(select) {
      // Keep only placeholder option if exists
      const placeholder = select.querySelector('option[value=""]');
      select.innerHTML = '';
      if (placeholder) {
        select.appendChild(placeholder);
      }
      select.value = '';

      // Update select2 if it's initialized
      if ($(select).hasClass('select2-hidden-accessible')) {
        $(select).trigger('change');
      } else {
        // Trigger change event for any dependent children
        select.dispatchEvent(new Event('change'));
      }
    }

    loadOptions(childName, parentValue, initialValue = null) {
      const childData = this.selects.get(childName);
      if (!childData) return;

      const { element, model, filterColumn, display, key } = childData;

      console.log('[ChainSelect] Loading options for', childName, {
        model,
        filterColumn,
        filterValue: parentValue,
        display,
        key,
        initialValue,
        isMultiple: Array.isArray(parentValue)
      });

      // Show loading state
      element.disabled = true;
      const loadingOption = document.createElement('option');
      loadingOption.textContent = 'Loading...';
      element.appendChild(loadingOption);

      // Make AJAX request
      fetch('/api/chain-select', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
          Accept: 'application/json'
        },
        body: JSON.stringify({
          model: model,
          filterColumn: filterColumn,
          filterValue: parentValue, // Can be single value or array
          display: display,
          key: key
        })
      })
        .then(response => {
          if (!response.ok) {
            throw new Error('Network response was not ok');
          }
          return response.json();
        })
        .then(data => {
          console.log('[ChainSelect] Received', data.items?.length || 0, 'items for', childName);
          this.populateSelect(element, data.items, key, display, initialValue);
          element.disabled = false;
        })
        .catch(error => {
          console.error('[ChainSelect] Error loading options:', error);
          this.clearSelect(element);
          const errorOption = document.createElement('option');
          errorOption.textContent = 'Error loading options';
          element.appendChild(errorOption);
          element.disabled = true;
        });
    }

    populateSelect(select, items, key, display, initialValue = null) {
      // Keep placeholder if exists (only for non-multiple selects)
      const placeholder = select.querySelector('option[value=""]');
      select.innerHTML = '';
      if (placeholder && !select.multiple) {
        select.appendChild(placeholder);
      }

      // Handle initial value as array for multiple select
      const initialValues = Array.isArray(initialValue) ? initialValue : initialValue ? [initialValue] : [];

      items.forEach(item => {
        const option = document.createElement('option');
        option.value = item[key];
        option.textContent = item[display];

        // Set selected if this matches the initial value(s)
        if (initialValues.length > 0 && initialValues.some(val => String(item[key]) === String(val))) {
          option.selected = true;
        }

        select.appendChild(option);
      });

      // Update select2 if it's initialized
      if ($(select).hasClass('select2-hidden-accessible')) {
        $(select).trigger('change');
      }

      // Only trigger change for dependent children if NOT during initial load
      const hasSelectedValue = select.multiple ? Array.from(select.selectedOptions).length > 0 : !!select.value;

      if (hasSelectedValue && !this.isLoadingInitial) {
        console.log('[ChainSelect] Triggering change for dependent children of', select.name);
        if ($(select).hasClass('select2-hidden-accessible')) {
          $(select).trigger('change');
        } else {
          select.dispatchEvent(new Event('change'));
        }
      }
    }

    loadInitialValues() {
      console.log('[ChainSelect] Loading initial values for edit mode');

      // Set flag to prevent clearing during initial load
      this.isLoadingInitial = true;

      // Process selects in order of dependency (parents first)
      const processedSelects = new Set();
      const loadPromises = [];

      const processSelect = selectName => {
        if (processedSelects.has(selectName)) return;

        const selectData = this.selects.get(selectName);
        if (!selectData) return;

        const { element, dependsOn, initialValue } = selectData;

        // If this select depends on another, process the parent first
        if (dependsOn) {
          processSelect(dependsOn);

          // Now check if parent has a value
          const parentSelect = document.querySelector(`select[name="${dependsOn}"], select[name="${dependsOn}[]"]`);
          if (parentSelect && initialValue) {
            // Get parent value(s)
            const parentValue = parentSelect.multiple
              ? Array.from(parentSelect.selectedOptions)
                  .map(opt => opt.value)
                  .filter(v => v !== '')
              : parentSelect.value;

            const hasParentValue = Array.isArray(parentValue) ? parentValue.length > 0 : !!parentValue;

            if (hasParentValue) {
              console.log('[ChainSelect] Loading initial data for', selectName, 'with value', initialValue);
              this.loadOptions(selectName, parentValue, initialValue);
            }
          }
        }

        processedSelects.add(selectName);
      };

      // Process all selects
      this.selects.forEach((data, name) => {
        if (data.initialValue) {
          processSelect(name);
        }
      });

      // Reset the flag after a short delay to allow all AJAX requests to complete
      setTimeout(() => {
        this.isLoadingInitial = false;
        console.log('[ChainSelect] Initial loading complete, enabling change handlers');
      }, 1000);
    }
  }

  // Initialize when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      new ChainSelect();
    });
  } else {
    new ChainSelect();
  }

  // Export for manual initialization if needed
  window.ChainSelect = ChainSelect;
})();
