<?php

namespace App\View\Components\Form;

use Illuminate\View\Component;
use Illuminate\Support\Collection;

/**
 * Dynamic Select component with chain/cascading support
 *
 * Usage examples (in Blade):
 * <x-form.select name="department_id" model="App\Http\Models\CmsDepartment" :query="[['where',['active',1]],['orderBy',['name','asc']]]" display="name" key="id" placeholder="Choose..." />
 *
 * Or pass items directly:
 * <x-form.select name="country" :items="$countries" display="label" key="code" />
 *
 * Query format: array of entries where each entry is [methodName, paramsArray]
 * Example: ['where',['status',1]] or ['orderBy', ['name','asc']]
 */
class Select extends Component
{
  public string $name;
  public string $display;
  public string $key;
  public $items; // Collection or array
  public $placeholder;
  public bool $multiple;
  public $value;
  public $dependsOn;
  public $filterColumn;
  public $model;
  public bool $ajax;
  public ?int $ajaxLimit;

  /**
   * Create the component instance.
   *
   * @param string $name
   * @param string|null $model Fully-qualified model class name (optional)
   * @param array|\Illuminate\Support\Collection|null $items Provide items directly (optional)
   * @param array|null $query Array of query operations to apply to the model builder
   * @param string $display Column to display
   * @param string $key Column to use as option value
   * @param string|null $placeholder Optional placeholder/empty option label
   * @param mixed $value Selected value(s)
   * @param bool $multiple Allow multiple selection
   * @param string|null $dependsOn Name of parent select (for chain selection)
   * @param string|null $filterColumn Database column to filter by parent value
   * @param bool $ajax Enable AJAX search for large datasets
   * @param int|null $ajaxLimit Limit number of results for AJAX search (default: 20)
   */
  public function __construct(
    string $name,
    ?string $model = null,
    $items = null,
    ?array $query = null,
    string $display = 'name',
    string $key = 'id',
    ?string $placeholder = null,
    $value = null,
    bool $multiple = false,
    $dependsOn = null,
    $filterColumn = null,
    bool $ajax = false,
    ?int $ajaxLimit = 20
  ) {
    $this->name = $name;
    $this->display = $display;
    $this->key = $key;
    $this->placeholder = $placeholder;
    $this->multiple = $multiple;
    $this->dependsOn = $dependsOn;
    $this->filterColumn = $filterColumn;
    $this->model = $model;
    $this->ajax = $ajax;
    $this->ajaxLimit = $ajaxLimit;

    // If AJAX search is enabled, start empty (items loaded via AJAX)
    if ($ajax) {
      // Only load selected items for edit mode
      if ($value && $model && class_exists($model)) {
        try {
          if (is_array($value)) {
            $this->items = $model::whereIn($key, $value)->get();
          } else {
            $selectedItem = $model::where($key, $value)->first();
            $this->items = $selectedItem ? collect([$selectedItem]) : collect();
          }
        } catch (\Throwable $e) {
          $this->items = collect();
        }
      } else {
        $this->items = collect();
      }
      $this->value = $value;
      return;
    }

    // If this select depends on another, start empty (populated via JS)
    if ($dependsOn) {
      // If a value is provided (edit mode), load the items for that value
      if ($value && $model && class_exists($model) && $filterColumn) {
        // We need the parent value to filter, but we don't have it here
        // So we'll load just the selected item to display
        try {
          $selectedItem = $model::where($key, $value)->first();
          if ($selectedItem) {
            $this->items = collect([$selectedItem]);
          } else {
            $this->items = collect();
          }
        } catch (\Throwable $e) {
          $this->items = collect();
        }
      } else {
        $this->items = collect();
      }
      $this->value = $value;
      return;
    }

    // Determine items: direct items take precedence
    if ($items instanceof Collection || is_array($items)) {
      $this->items = collect($items);
    } elseif ($model && class_exists($model)) {
      // Build query using provided query array
      $builder = $model::query();

      if (is_array($query)) {
        foreach ($query as $q) {
          // support formats: ['method', ['param1', 'param2']] or ['method', param]
          if (!is_array($q)) {
            continue;
          }

          $method = $q[0] ?? null;
          $params = $q[1] ?? [];

          if ($method && is_string($method)) {
            if (!is_array($params)) {
              $params = [$params];
            }

            // call the method on builder; some methods (select) return builder, others modify it
            try {
              $result = call_user_func_array([$builder, $method], $params);
              if ($result instanceof \Illuminate\Database\Query\Builder || $result instanceof \Illuminate\Database\Eloquent\Builder) {
                $builder = $result;
              }
            } catch (\Throwable $e) {
              // ignore invalid query parts, but don't break whole component
              continue;
            }
          }
        }
      }

      $this->items = $builder->get();
    } else {
      $this->items = collect();
    }

    $this->value = $value;
  }

  /**
   * Get the view/contents that represent the component.
   *
   * @return \Illuminate\View\View|string
   */
  public function render()
  {
    return view('components.form.select', [
      'items' => $this->items,
      'name' => $this->name,
      'display' => $this->display,
      'key' => $this->key,
      'placeholder' => $this->placeholder,
      'multiple' => $this->multiple,
      'value' => $this->value,
      'dependsOn' => $this->dependsOn,
      'filterColumn' => $this->filterColumn,
      'model' => $this->model,
      'ajax' => $this->ajax,
      'ajaxLimit' => $this->ajaxLimit,
    ]);
  }
}
