<?php

namespace App\Traits;

use Illuminate\Support\Facades\Crypt;
use Illuminate\Contracts\Encryption\DecryptException;

trait EncryptsIds
{
    /**
     * Encrypt an ID for public use
     *
     * @param int|string $id
     * @return string
     */
    public function encryptId($id)
    {
        if (empty($id)) {
            return null;
        }

        try {
            return Crypt::encryptString((string)$id);
        } catch (\Exception $e) {
            \Log::error('ID Encryption Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Decrypt an ID from encrypted string
     *
     * @param string $encryptedId
     * @return int|null
     */
    public function decryptId($encryptedId)
    {
        if (empty($encryptedId)) {
            return null;
        }

        try {
            $decrypted = Crypt::decryptString($encryptedId);
            return is_numeric($decrypted) ? (int)$decrypted : null;
        } catch (DecryptException $e) {
            \Log::warning('ID Decryption Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get encrypted ID attribute
     *
     * @return string|null
     */
    public function getEncryptedIdAttribute()
    {
        return $this->encryptId($this->id);
    }

    /**
     * Encrypt multiple IDs
     *
     * @param array $ids
     * @return array
     */
    public function encryptIds(array $ids)
    {
        return array_map(function ($id) {
            return $this->encryptId($id);
        }, $ids);
    }

    /**
     * Decrypt multiple IDs
     *
     * @param array $encryptedIds
     * @return array
     */
    public function decryptIds(array $encryptedIds)
    {
        return array_filter(array_map(function ($encryptedId) {
            return $this->decryptId($encryptedId);
        }, $encryptedIds));
    }
}
