<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Helpers\IdEncryption;
use Symfony\Component\HttpFoundation\Response;

class DecryptRouteIds
{
    /**
     * Route parameters that should be decrypted
     *
     * @var array
     */
    protected $encryptedParameters = [
        'id',
        'studentId',
        'challanId',
        'classId',
        'sectionId',
        'staffId',
        'schoolId',
    ];

    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Get all route parameters
        $routeParameters = $request->route()->parameters();

        foreach ($routeParameters as $key => $value) {
            // Check if this parameter should be decrypted
            if (in_array($key, $this->encryptedParameters) && !empty($value)) {
                // Decrypt the ID
                $decryptedId = IdEncryption::decrypt($value);

                if ($decryptedId === null) {
                    // If decryption fails, check if it's already a numeric ID (backwards compatibility)
                    if (is_numeric($value)) {
                        // Allow numeric IDs for backwards compatibility during transition
                        continue;
                    }

                    // Invalid encrypted ID
                    abort(404, 'Invalid or corrupted resource identifier');
                }

                // Replace the encrypted ID with decrypted one
                $request->route()->setParameter($key, $decryptedId);
            }
        }

        return $next($request);
    }

    /**
     * Add custom parameter to be decrypted
     *
     * @param string $parameter
     * @return void
     */
    public function addEncryptedParameter(string $parameter)
    {
        if (!in_array($parameter, $this->encryptedParameters)) {
            $this->encryptedParameters[] = $parameter;
        }
    }
}
