<?php

namespace App\Http\Controllers\myapp;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Log;
use App\Models\App_student;
use App\Models\App_school;
use App\Models\App_staff;
use App\Models\App_stype;
use App\Models\App_class;
use App\Models\App_section;
use App\Models\FeeChallans;
use App\Models\App_fee_extra;
use App\Helpers\IdEncryption;

class Admin extends Controller
{
  /**
   * Get status label helper function
   */
  private function getStatusLabel($status)
  {
    $labels = [
      0 => 'Issued',
      1 => 'Pending',
      2 => 'Waiting Approval',
      3 => 'Cancelled',
      4 => 'Paid'
    ];
    return $labels[$status] ?? 'Unknown';
  }

  /**
   * Check if user can create challans
   */
  private function canCreateChallan()
  {
    $user = Auth::user();
    $staffTypeId = $user->app_stype_id ?? null;

    // app_stype_id = 2 (Cashier) and 3 (Viewer) cannot create challans
    if (in_array($staffTypeId, [2, 3])) {
      return false;
    }

    return true;
  }

  /**
   * Check if user can edit challan
   */
  private function canEditChallan($challan)
  {
    $user = Auth::user();
    $staffTypeId = $user->app_stype_id ?? null;

    // app_stype_id = 3 (Viewer) cannot edit any challan
    if ($staffTypeId == 3) {
      return false;
    }

    // app_stype_id = 2 (Cashier) can only edit Issued (0) and Pending (1) challans
    // Cannot edit Paid (4) or Cancelled (3) challans
    if ($staffTypeId == 2) {
      if (!in_array($challan->status, [0, 1])) {
        return false;
      }
    }

    return true;
  }

  /**
   * Get allowed status transitions for user
   */
  private function getAllowedStatusTransitions($currentStatus)
  {
    $user = Auth::user();
    $staffTypeId = $user->app_stype_id ?? null;

    // app_stype_id = 2 (Cashier) can only mark as Waiting Approval (2)
    if ($staffTypeId == 2) {
      if (in_array($currentStatus, [0, 1])) {
        return [2]; // Can only change to Waiting Approval
      }
      return []; // Cannot change status
    }

    // Default: admin can change to any status
    return [0, 1, 2, 3, 4];
  }

  public function index()
  {
    return view('myapp.admin.dashboard');
  }


  public function empty()
  {
    return view('myapp.admin.empty');
  }


  public function cases()
  {
    return view('myapp.admin.form');
  }


  // public function test()
  // {
  //   return view('myapp.example');
  // }

  public function test()
  {
    return view('myapp.form-components-example');
  }


  // public function test()
  // {
  //   return view('myapp.admin.empty');
  // }



  /**
   * Show the student creation form.
   */
  public function studentCreate()
  {
    $schoolId = Auth::user()->app_school_id ?? null;
    $nextRollNo = null;
    if ($schoolId) {
      $school = App_school::find($schoolId);
      $nextRollNo = ($school->roll_no ?? 0) + 1;
    }

    // Check if this is an AJAX request for modal
    if (request()->ajax() || request()->wantsJson()) {
      return view('myapp.admin.student_form', compact('nextRollNo'));
    }

    return view('myapp.admin.student_create', compact('nextRollNo'));
  }



  /**
   * Store a new student.
   */
  public function studentStore(Request $request)
  {
    $data = $request->validate([
      'roll_no' => 'nullable|string|max:100',
      'name' => 'required|string|max:255',
      'fname' => 'nullable|string|max:255',
      'mob1' => 'nullable|string|max:50',
      'mob2' => 'nullable|string|max:50',
      'join_date' => 'nullable|date',
      'address' => 'nullable|string',
      'app_class_id' => 'nullable|integer',
      'app_section_id' => 'nullable|integer',
      'fee' => 'nullable|numeric',
      'fstatus' => 'nullable|in:0,1',
      'image' => 'nullable|image|max:2048',
      'image_data' => 'nullable|string',

      // app_school_id will be taken from authenticated user
    ]);

    // Ensure school id is taken from authenticated user
    $schoolId = $request->user()->app_school_id ?? $request->input('app_school_id') ?? null;
    if ($schoolId) {
      $data['app_school_id'] = $schoolId;
    }

    // Ensure staff id is taken from authenticated user or request; fall back to authenticated user id
    $staffId = $request->user()->app_staff_id ?? $request->input('app_staff_id') ?? $request->user()->id ?? null;
    // Always set the key so the create call includes it (avoids DB missing-default error)
    if (!array_key_exists('app_staff_id', $data) || $data['app_staff_id'] !== $staffId) {
      $data['app_staff_id'] = $staffId;
    }

    // Handle uploaded file: store under storage/app/public/imgdata/... so it's accessible via public/storage
    if ($request->hasFile('image')) {
      $file = $request->file('image');
      $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
      $extension = $file->getClientOriginalExtension();
      $filename = time() . '_' . preg_replace('/[^A-Za-z0-9_\-]/', '_', $originalName) . '.' . $extension;

      Storage::disk('public')->putFileAs('imgdata/' . $schoolId . '/students', $file, $filename);
      $data['image'] = 'storage/imgdata/' . $schoolId . '/students/' . $filename;
    } elseif ($request->filled('image_data')) {
      // handle base64 image from webcam and save into storage/app/public
      $imageData = $request->input('image_data');
      if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
        $imageData = substr($imageData, strpos($imageData, ',') + 1);
        $type = strtolower($type[1]); // jpg, png, etc
        $imageData = base64_decode($imageData);
        if ($imageData !== false) {
          $filename = time() . '_camera.' . $type;
          $path = 'imgdata/' . $schoolId . '/students/' . $filename;
          Storage::disk('public')->put($path, $imageData);
          $data['image'] = 'storage/' . $path;
        }
      }
    }

    $student = null;

    if (!empty($schoolId)) {
      // Use a transaction and row lock to avoid race conditions when generating roll numbers
      DB::transaction(function () use (&$student, $data, $schoolId) {
        $school = App_school::lockForUpdate()->find($schoolId);
        $next = ($school->roll_no ?? 0) + 1;

        // If a roll_no was provided from form and is greater than computed next, honor it
        if (empty($data['roll_no'])) {
          $data['roll_no'] = $next;
        } else {
          $next = max($next, $data['roll_no']);
        }

        $student = App_student::create($data);

        // Persist the updated roll_no back to the school
        $school->roll_no = $next;
        $school->save();
      });
    } else {
      $student = App_student::create($data);
    }

    if ($request->ajax() || $request->wantsJson()) {
      return response()->json([
        'success' => true,
        'message' => 'Student created successfully',
        'student_id' => $student->id
      ]);
    }

    return redirect()->route('admin.students.create')->with('success', 'Student created successfully.');
  }



  /**
   * Show the student list page.
   */
  public function studentlist()
  {
    $schoolId = Auth::user()->app_school_id ?? null;
    $nextRollNo = null;
    $classes = [];

    if ($schoolId) {
      $school = App_school::find($schoolId);
      $nextRollNo = ($school->roll_no ?? 0) + 1;
      // Get all classes for this school
      $classes = App_class::where('app_school_id', $schoolId)->get();
    }

    return view('myapp.admin.student_list', compact('nextRollNo', 'classes'));
  }

  /**
   * Get student data for DataTables (server-side processing).
   */
  public function studentListData(Request $request)
  {
    try {
      $schoolId = $request->user()->app_school_id;

      if (!$schoolId) {
        return response()->json([
          'draw' => intval($request->input('draw')),
          'recordsTotal' => 0,
          'recordsFiltered' => 0,
          'data' => []
        ]);
      }

      // Base query with relationships
      $query = App_student::with(['class', 'section'])
        ->where('app_students.app_school_id', $schoolId)
        ->where('app_students.status', 0)
        ->select([
          'app_students.id',
          'app_students.name',
          'app_students.fname',
          'app_students.mob1',
          'app_students.mob2',
          'app_students.join_date',
          'app_students.image',
          'app_students.roll_no',
          'app_students.fstatus',
          'app_students.app_class_id',
          'app_students.app_section_id'
        ]);

      // Get total count before filtering
      $recordsTotal = App_student::where('app_school_id', $schoolId)
        ->where('status', 0)
        ->count();

      // Apply search filter
      $search = $request->input('search.value');
      if (!empty($search)) {
        $query->where(function ($q) use ($search) {
          $searchLower = strtolower($search);

          $q->where('app_students.name', 'like', "%{$search}%")
            ->orWhere('app_students.fname', 'like', "%{$search}%")
            ->orWhere('app_students.roll_no', 'like', "%{$search}%")
            ->orWhere('app_students.mob1', 'like', "%{$search}%")
            ->orWhere('app_students.mob2', 'like', "%{$search}%")
            ->orWhereHas('class', function ($classQuery) use ($search) {
              $classQuery->where('name', 'like', "%{$search}%");
            })
            ->orWhereHas('section', function ($sectionQuery) use ($search) {
              $sectionQuery->where('name', 'like', "%{$search}%");
            });

          // Status keyword search: map "regular"/"paid" to fstatus = 0, "free" to fstatus = 1
          if (stripos($searchLower, 'regular') !== false || stripos($searchLower, 'paid') !== false) {
            $q->orWhere('app_students.fstatus', 0);
          }
          if (stripos($searchLower, 'free') !== false) {
            $q->orWhere('app_students.fstatus', 1);
          }
        });
      }

      // Apply class filter
      $classId = $request->input('class_id');
      if (!empty($classId)) {
        $query->where('app_students.app_class_id', $classId);
      }

      // Apply section filter
      $sectionId = $request->input('section_id');
      if (!empty($sectionId)) {
        $query->where('app_students.app_section_id', $sectionId);
      }

      // Get filtered count
      $recordsFiltered = $query->count();

      // Define columns mapping (must match DataTable column order)
      $columns = ['', '', 'roll_no', 'name', 'fname', 'mob1', 'join_date', '', 'fstatus', '', ''];

      // Apply ordering - Default to sort by roll_no DESC
      $orderColumnIndex = $request->input('order.0.column', 2);
      $orderDir = $request->input('order.0.dir', 'asc');

      // For roll_no (column 2), force DESC order
      if ($orderColumnIndex == 2) {
        $orderDir = 'DESC';
      } else {
        $orderDir = strtoupper($orderDir);
      }

      Log::debug('DataTable Order Parameters', [
        'orderColumnIndex' => $orderColumnIndex,
        'orderDir' => $orderDir,
        'columnName' => isset($columns[$orderColumnIndex]) ? $columns[$orderColumnIndex] : 'unknown'
      ]);

      if (isset($columns[$orderColumnIndex]) && !empty($columns[$orderColumnIndex])) {
        $query->orderBy('app_students.' . $columns[$orderColumnIndex], $orderDir);
      } else {
        $query->orderBy('app_students.roll_no', 'DESC');
      }

      // Apply pagination
      $start = $request->input('start', 0);
      $length = $request->input('length', 10);
      $query->offset($start)->limit($length);

      // Get results
      $students = $query->get();

      // Format data for DataTables
      $data = [];
      foreach ($students as $student) {
        $encryptedId = Crypt::encryptString($student->id);

        // Format image path
        $imagePath = $student->image
          ? asset($student->image)
          : asset('assets/img/avatars/default-avatar.png');

        Log::debug('Student image path', [
          'student_id' => $student->id,
          'image_raw' => $student->image,
          'image_path' => $imagePath
        ]);

        // Format mobile numbers
        $mobileDisplay = $student->mob1 ?? '';
        if ($student->mob2) {
          $mobileDisplay .= ' / ' . $student->mob2;
        }

        // Format class/section
        $classSection = '';
        if ($student->class) {
          $classSection = $student->class->name;
        }
        if ($student->section) {
          $classSection .= ' / ' . $student->section->name;
        }

        // Format fee status badge
        $feeStatusBadge = $student->fstatus == 0
          ? '<span class="badge rounded-pill bg-label-success">Regular</span>'
          : '<span class="badge rounded-pill bg-label-danger">Free</span>';

        $data[] = [
          'id' => $student->id,
          'DT_RowId' => 'row_' . $student->id,
          'roll_no' => $student->roll_no ?? '',
          'name' => $student->name,
          'fname' => $student->fname ?? '',
          'mobile' => $mobileDisplay,
          'join_date' => $student->join_date ? date('d-m-Y', strtotime($student->join_date)) : '',
          'class_section' => $classSection,
          'fstatus' => $feeStatusBadge,
          'image' => $imagePath,
          'encrypted_id' => $encryptedId
        ];
      }

      return response()->json([
        'draw' => intval($request->input('draw')),
        'recordsTotal' => $recordsTotal,
        'recordsFiltered' => $recordsFiltered,
        'data' => $data
      ]);
    } catch (\Exception $e) {
      Log::error('studentListData error: ' . $e->getMessage());
      return response()->json([
        'draw' => intval($request->input('draw')),
        'recordsTotal' => 0,
        'recordsFiltered' => 0,
        'data' => [],
        'error' => $e->getMessage()
      ], 500);
    }
  }

  /**
   * Show the student edit form.
   */
  public function studentEdit($id)
  {
    try {
      $studentId = Crypt::decryptString($id);
      $schoolId = Auth::user()->app_school_id;

      $student = App_student::where('id', $studentId)
        ->where('app_school_id', $schoolId)
        ->firstOrFail();

      // Check if this is an AJAX request for modal
      if (request()->ajax() || request()->wantsJson()) {
        return view('myapp.admin.student_edit_form', compact('student'));
      }

      return view('myapp.admin.student_edit', compact('student'));
    } catch (\Exception $e) {
      if (request()->ajax()) {
        return response()->json(['error' => 'Student not found'], 404);
      }
      return redirect()->route('admin.students.list')->with('error', 'Student not found');
    }
  }

  /**
   * Update the student.
   */
  public function studentUpdate(Request $request, $id)
  {
    try {
      // Log everything about the request
      $allInput = $request->all();
      $inputKeys = array_keys($allInput);

      Log::info('Update request received - RAW DATA', [
        'id' => $id,
        'method' => $request->method(),
        'content_type' => $request->header('content-type'),
        'has_name' => isset($allInput['name']),
        'name_value' => $allInput['name'] ?? 'NOT SET',
        'name_is_empty' => empty($allInput['name']),
        'name_length' => strlen($allInput['name'] ?? ''),
        'all_input_count' => count($allInput),
        'input_keys' => $inputKeys,
      ]);

      // Log each field
      foreach ($allInput as $key => $value) {
        $displayValue = (is_string($value) && strlen($value) > 100) ? substr($value, 0, 100) . '...' : $value;
        Log::debug("Request field: {$key}", ['value' => $displayValue, 'type' => gettype($value)]);
      }

      $studentId = Crypt::decryptString($id);
      $schoolId = $request->user()->app_school_id;

      $student = App_student::where('id', $studentId)
        ->where('app_school_id', $schoolId)
        ->firstOrFail();

      $data = $request->validate([
        'roll_no' => 'nullable|string|max:100',
        'name' => 'required|string|max:255',
        'fname' => 'nullable|string|max:255',
        'mob1' => 'nullable|string|max:50',
        'mob2' => 'nullable|string|max:50',
        'join_date' => 'nullable|date',
        'address' => 'nullable|string',
        'app_class_id' => 'nullable|integer',
        'app_section_id' => 'nullable|integer',
        'fee' => 'nullable|numeric',
        'fstatus' => 'nullable|in:0,1',
        'image' => 'nullable|image|max:2048',
        'image_data' => 'nullable|string',
      ]);

      Log::info('Validation passed', ['data' => $data]);

      // Handle uploaded file
      if ($request->hasFile('image')) {
        // Delete old image if exists
        if ($student->image && file_exists(public_path($student->image))) {
          unlink(public_path($student->image));
        }

        $file = $request->file('image');
        $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $extension = $file->getClientOriginalExtension();
        $filename = time() . '_' . preg_replace('/[^A-Za-z0-9_\-]/', '_', $originalName) . '.' . $extension;

        Storage::disk('public')->putFileAs('imgdata/' . $schoolId . '/students', $file, $filename);
        $data['image'] = 'storage/imgdata/' . $schoolId . '/students/' . $filename;
      } elseif ($request->filled('image_data')) {
        // Handle base64 image from webcam
        $imageData = $request->input('image_data');
        if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
          // Delete old image if exists
          if ($student->image && file_exists(public_path($student->image))) {
            unlink(public_path($student->image));
          }

          $imageData = substr($imageData, strpos($imageData, ',') + 1);
          $type = strtolower($type[1]);
          $imageData = base64_decode($imageData);
          if ($imageData !== false) {
            $filename = time() . '_camera.' . $type;
            $path = 'imgdata/' . $schoolId . '/students/' . $filename;
            Storage::disk('public')->put($path, $imageData);
            $data['image'] = 'storage/' . $path;
          }
        }
      }

      $student->update($data);
      Log::info('Student updated successfully', ['student_id' => $studentId, 'school_id' => $schoolId]);

      if ($request->ajax() || $request->wantsJson()) {
        // Reload fresh student data with relationships
        $updatedStudent = App_student::with(['class', 'section'])->find($studentId);

        // Format the row data (same format as in studentListData)
        $imagePath = $updatedStudent->image
          ? asset($updatedStudent->image)
          : asset('assets/img/avatars/default-avatar.png');

        $mobileDisplay = $updatedStudent->mob1 ?? '';
        if ($updatedStudent->mob2) {
          $mobileDisplay .= ' / ' . $updatedStudent->mob2;
        }

        $classSection = '';
        if ($updatedStudent->class) {
          $classSection = $updatedStudent->class->name;
        }
        if ($updatedStudent->section) {
          $classSection .= ' / ' . $updatedStudent->section->name;
        }

        $feeStatusBadge = $updatedStudent->fstatus == 0
          ? '<span class="badge rounded-pill bg-label-success">Regular</span>'
          : '<span class="badge rounded-pill bg-label-danger">Free</span>';

        $rowData = [
          'id' => $updatedStudent->id,
          'roll_no' => $updatedStudent->roll_no ?? '',
          'name' => $updatedStudent->name,
          'fname' => $updatedStudent->fname ?? '',
          'mobile' => $mobileDisplay,
          'join_date' => $updatedStudent->join_date ? date('d-m-Y', strtotime($updatedStudent->join_date)) : '',
          'class_section' => $classSection,
          'fstatus' => $feeStatusBadge,
          'image' => $imagePath,
          'encrypted_id' => Crypt::encryptString($updatedStudent->id)
        ];

        return response()->json([
          'success' => true,
          'message' => 'Student updated successfully',
          'student_id' => $studentId,
          'row_data' => $rowData
        ]);
      }

      return redirect()->route('admin.students.list')->with('success', 'Student updated successfully');
    } catch (\Illuminate\Validation\ValidationException $e) {
      Log::error('Validation error updating student', ['errors' => $e->errors(), 'id' => $id]);
      if ($request->ajax()) {
        return response()->json(['error' => 'Validation failed', 'errors' => $e->errors()], 422);
      }
      return redirect()->back()->withErrors($e->errors())->withInput();
    } catch (\Exception $e) {
      Log::error('Error updating student', ['message' => $e->getMessage(), 'code' => $e->getCode(), 'id' => $id]);
      if ($request->ajax()) {
        return response()->json(['error' => $e->getMessage() ?? 'Failed to update student'], 500);
      }
      return redirect()->back()->with('error', $e->getMessage() ?? 'Failed to update student');
    }
  }

  /**
   * Show student profile.
   */
  public function studentProfile($id)
  {
    try {
      $studentId = Crypt::decryptString($id);
      $schoolId = Auth::user()->app_school_id;

      $student = App_student::with(['class', 'section', 'school'])
        ->where('id', $studentId)
        ->where('app_school_id', $schoolId)
        ->firstOrFail();

      return view('myapp.admin.student_profile', compact('student'));
    } catch (\Exception $e) {
      return redirect()->route('admin.students.list')->with('error', 'Student not found');
    }
  }

  /**
   * Get student profile as HTML for AJAX modal view (with payment history tab).
   */
  public function studentProfileModal($id)
  {
    try {
      $studentId = Crypt::decryptString($id);
      $schoolId = Auth::user()->app_school_id;

      $student = App_student::with(['class', 'section', 'school'])
        ->where('id', $studentId)
        ->where('app_school_id', $schoolId)
        ->firstOrFail();

      // Check if this is an AJAX request
      if (request()->ajax() || request()->wantsJson()) {
        return view('myapp.admin.student_profile_modal', compact('student'));
      }

      return view('myapp.admin.student_profile', compact('student'));
    } catch (\Exception $e) {
      Log::error('Error loading student profile modal: ' . $e->getMessage());
      return response()->json(['error' => 'Student not found'], 404);
    }
  }

  /**
   * Get sections for a given class.
   */
  public function getSectionsByClass($classId)
  {
    try {
      $schoolId = Auth::user()->app_school_id;

      // Get sections for the given class, filtered by school
      $sections = App_section::where('app_class_id', $classId)
        ->whereHas('class', function ($query) use ($schoolId) {
          $query->where('app_school_id', $schoolId);
        })
        ->select('id', 'name')
        ->get();

      return response()->json($sections);
    } catch (\Exception $e) {
      Log::error('Error fetching sections:', ['error' => $e->getMessage()]);
      return response()->json([], 500);
    }
  }

  /**
   * Show the student fee management page.
   */
  public function studentManageFee(Request $request)
  {
    $schoolId = Auth::user()->app_school_id ?? null;
    $staffTypeId = Auth::user()->app_stype_id ?? null;
    $canCreate = $this->canCreateChallan();
    $nextRollNo = null;
    $classes = [];

    if ($schoolId) {
      $school = App_school::find($schoolId);
      $nextRollNo = ($school->roll_no ?? 0) + 1;
      // Get all classes for this school
      $classes = App_class::where('app_school_id', $schoolId)->get();
    }

    return view('myapp.admin.student_managefee', compact('nextRollNo', 'classes', 'staffTypeId', 'canCreate'));
  }

  /**
   * Get student payment history for modal (AJAX).
   */
  public function studentPaymentHistory($studentId)
  {
    try {
      $schoolId = Auth::user()->app_school_id ?? null;

      if (!$schoolId) {
        return response()->json(['error' => 'School not found'], 404);
      }

      $student = App_student::with(['class', 'section'])
        ->where('id', $studentId)
        ->where('app_school_id', $schoolId)
        ->first();

      if (!$student) {
        return response()->json(['error' => 'Student not found'], 404);
      }

      $challans = FeeChallans::with(['class', 'section', 'history.staff'])
        ->where('app_student_id', $studentId)
        ->where('app_school_id', $schoolId)
        ->orderBy('created_at', 'desc')
        ->get();

      return view('myapp.admin.partials.student_payment_history_modal', compact('student', 'challans'));
    } catch (\Exception $e) {
      Log::error('studentPaymentHistory error: ' . $e->getMessage());
      return response('<div class="alert alert-danger">Error loading payment history</div>', 500);
    }
  }

  /**
   * Get student data for Fee Management DataTables (server-side processing).
   */
  public function studentManageFeeData(Request $request)
  {
    try {
      $schoolId = $request->user()->app_school_id;

      if (!$schoolId) {
        return response()->json([
          'draw' => intval($request->input('draw')),
          'recordsTotal' => 0,
          'recordsFiltered' => 0,
          'data' => []
        ]);
      }

      // Base query with relationships
      $query = App_student::with(['class', 'section'])
        ->where('app_students.app_school_id', $schoolId)
        ->where('app_students.status', 0)
        ->select([
          'app_students.id',
          'app_students.name',
          'app_students.fname',
          'app_students.mob1',
          'app_students.mob2',
          'app_students.join_date',
          'app_students.image',
          'app_students.roll_no',
          'app_students.fstatus',
          'app_students.app_class_id',
          'app_students.app_section_id'
        ]);

      // Get total count before filtering
      $recordsTotal = App_student::where('app_school_id', $schoolId)
        ->where('status', 0)
        ->count();

      // Apply search filter
      $search = $request->input('search.value');
      if (!empty($search)) {
        $query->where(function ($q) use ($search) {
          $searchLower = strtolower($search);

          $q->where('app_students.name', 'like', "%{$search}%")
            ->orWhere('app_students.fname', 'like', "%{$search}%")
            ->orWhere('app_students.roll_no', 'like', "%{$search}%")
            ->orWhere('app_students.mob1', 'like', "%{$search}%")
            ->orWhere('app_students.mob2', 'like', "%{$search}%")
            ->orWhereHas('class', function ($classQuery) use ($search) {
              $classQuery->where('name', 'like', "%{$search}%");
            })
            ->orWhereHas('section', function ($sectionQuery) use ($search) {
              $sectionQuery->where('name', 'like', "%{$search}%");
            });

          // Status keyword search: map "regular"/"paid" to fstatus = 0, "free" to fstatus = 1
          if (stripos($searchLower, 'regular') !== false || stripos($searchLower, 'paid') !== false) {
            $q->orWhere('app_students.fstatus', 0);
          }
          if (stripos($searchLower, 'free') !== false) {
            $q->orWhere('app_students.fstatus', 1);
          }
        });
      }

      // Apply class filter
      $classId = $request->input('class_id');
      if (!empty($classId)) {
        $query->where('app_students.app_class_id', $classId);
      }

      // Apply section filter
      $sectionId = $request->input('section_id');
      if (!empty($sectionId)) {
        $query->where('app_students.app_section_id', $sectionId);
      }

      // Apply fee status filter
      $feeStatus = $request->input('fee_status');
      if (!empty($feeStatus)) {
        if ($feeStatus === 'pending') {
          // Students with pending challans (status 0, 1, 2)
          $query->whereHas('challans', function ($q) use ($schoolId) {
            $q->where('app_school_id', $schoolId)
              ->whereIn('status', [0, 1, 2]);
          });
        } elseif ($feeStatus === 'no_due') {
          // Students with NO pending challans
          $query->whereDoesntHave('challans', function ($q) use ($schoolId) {
            $q->where('app_school_id', $schoolId)
              ->whereIn('status', [0, 1, 2]);
          });
        }
      }

      // Get filtered count
      $recordsFiltered = $query->count();

      // Define columns mapping (must match DataTable column order)
      $columns = ['', '', 'roll_no', 'name', 'fname', 'mob1', 'join_date', '', 'fstatus', ''];

      // Apply ordering - Default to sort by roll_no DESC
      $orderColumnIndex = $request->input('order.0.column', 2);
      $orderDir = $request->input('order.0.dir', 'asc');

      // For roll_no (column 2), force DESC order
      if ($orderColumnIndex == 2) {
        $orderDir = 'DESC';
      } else {
        $orderDir = strtoupper($orderDir);
      }

      Log::debug('DataTable Order Parameters (Fee Management)', [
        'orderColumnIndex' => $orderColumnIndex,
        'orderDir' => $orderDir,
        'columnName' => isset($columns[$orderColumnIndex]) ? $columns[$orderColumnIndex] : 'unknown'
      ]);

      if (isset($columns[$orderColumnIndex]) && !empty($columns[$orderColumnIndex])) {
        $query->orderBy('app_students.' . $columns[$orderColumnIndex], $orderDir);
      } else {
        $query->orderBy('app_students.roll_no', 'DESC');
      }

      // Apply pagination
      $start = $request->input('start', 0);
      $length = $request->input('length', 10);
      $query->offset($start)->limit($length);

      // Get results
      $students = $query->get();

      // Format data for DataTables
      $data = [];
      foreach ($students as $student) {
        $encryptedId = Crypt::encryptString($student->id);

        // Format image path
        $imagePath = $student->image
          ? asset($student->image)
          : asset('assets/img/avatars/default-avatar.png');

        Log::debug('Student image path (Fee Management)', [
          'student_id' => $student->id,
          'image_raw' => $student->image,
          'image_path' => $imagePath
        ]);

        // Format mobile numbers
        $mobileDisplay = $student->mob1 ?? '';
        if ($student->mob2) {
          $mobileDisplay .= ' / ' . $student->mob2;
        }

        // Format class/section
        $classSection = '';
        if ($student->class) {
          $classSection = $student->class->name;
        }
        if ($student->section) {
          $classSection .= ' / ' . $student->section->name;
        }

        // Format fee status badge
        $feeStatusBadge = $student->fstatus == 0
          ? '<span class="badge rounded-pill bg-label-success">Regular</span>'
          : '<span class="badge rounded-pill bg-label-danger">Free</span>';

        // Calculate pending fee (status 0, 1, 2)
        $pendingChallans = FeeChallans::where('app_student_id', $student->id)
          ->where('app_school_id', $schoolId)
          ->whereIn('status', [0, 1, 2])
          ->get();

        $pendingCount = $pendingChallans->count();
        $pendingAmount = $pendingChallans->sum('amount');

        if ($pendingCount > 0) {
          $feeDisplay = '<span class="text-danger fw-bold">₨' . number_format($pendingAmount, 0) . '</span><br><small class="text-muted">' . $pendingCount . ' challan(s) pending</small>';
        } else {
          $feeDisplay = '<span class="text-success">No fee pending</span>';
        }

        $data[] = [
          'id' => $student->id,
          'DT_RowId' => 'row_' . $student->id,
          'roll_no' => $student->roll_no ?? '',
          'name' => $student->name,
          'fname' => $student->fname ?? '',
          'mobile' => $mobileDisplay,
          'join_date' => $student->join_date ? date('d-m-Y', strtotime($student->join_date)) : '',
          'class_section' => $classSection,
          'fstatus' => $feeStatusBadge,
          'pending_fee' => $feeDisplay,
          'image' => $imagePath,
          'encrypted_id' => $encryptedId
        ];
      }

      return response()->json([
        'draw' => intval($request->input('draw')),
        'recordsTotal' => $recordsTotal,
        'recordsFiltered' => $recordsFiltered,
        'data' => $data
      ]);
    } catch (\Exception $e) {
      Log::error('studentManageFeeData error: ' . $e->getMessage());
      return response()->json([
        'draw' => intval($request->input('draw')),
        'recordsTotal' => 0,
        'recordsFiltered' => 0,
        'data' => [],
        'error' => $e->getMessage()
      ], 500);
    }
  }

  /**
   * Show the fee challans list page.
   */
  public function challanList()
  {
    $schoolId = Auth::user()->app_school_id ?? null;
    $staffTypeId = Auth::user()->app_stype_id ?? null;
    $canCreate = $this->canCreateChallan();
    $classes = [];
    $stats = [
      'total_challans' => 0,
      'issued' => 0,
      'pending' => 0,
      'cancelled' => 0,
      'paid' => 0,
      'paid_amount' => 0,
      'unpaid_amount' => 0,
    ];

    if ($schoolId) {
      $classes = App_class::where('app_school_id', $schoolId)->get();

      // Calculate statistics
      $stats['total_challans'] = FeeChallans::where('app_school_id', $schoolId)->count();
      $stats['issued'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 0)->count();
      $stats['pending'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 1)->count();
      $stats['cancelled'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 3)->count();
      $stats['paid'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 4)->count();
      $stats['paid_amount'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 4)->sum('amount');
      $stats['unpaid_amount'] = FeeChallans::where('app_school_id', $schoolId)->whereIn('status', [0, 1, 2])->sum('amount');
    }

    return view('myapp.admin.challan_list', compact('classes', 'stats', 'staffTypeId', 'canCreate'));
  }

  /**
   * Get challan statistics via API (for dynamic refresh).
   */
  public function challanStats(Request $request)
  {
    try {
      $schoolId = $request->user()->app_school_id ?? null;

      $stats = [
        'total_challans' => 0,
        'issued' => 0,
        'pending' => 0,
        'cancelled' => 0,
        'paid' => 0,
        'paid_amount' => 0,
        'unpaid_amount' => 0,
      ];

      if ($schoolId) {
        $stats['total_challans'] = FeeChallans::where('app_school_id', $schoolId)->count();
        $stats['issued'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 0)->count();
        $stats['pending'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 1)->count();
        $stats['cancelled'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 3)->count();
        $stats['paid'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 4)->count();
        $stats['paid_amount'] = FeeChallans::where('app_school_id', $schoolId)->where('status', 4)->sum('amount');
        $stats['unpaid_amount'] = FeeChallans::where('app_school_id', $schoolId)->whereIn('status', [0, 1, 2])->sum('amount');
      }

      return response()->json($stats);
    } catch (\Exception $e) {
      Log::error('challanStats error: ' . $e->getMessage());
      return response()->json(['error' => $e->getMessage()], 500);
    }
  }

  /**
   * Get fee challans data for DataTables (server-side processing).
   */
  public function challanListData(Request $request)
  {
    try {
      $schoolId = $request->user()->app_school_id;

      // Debug logging
      Log::info('challanListData - User School ID: ' . $schoolId);
      Log::info('challanListData - Total records in app_fee: ' . FeeChallans::count());
      Log::info('challanListData - Records with school_id=' . $schoolId . ': ' . FeeChallans::where('app_school_id', $schoolId)->count());

      if (!$schoolId) {
        return response()->json([
          'draw' => intval($request->input('draw')),
          'recordsTotal' => 0,
          'recordsFiltered' => 0,
          'data' => []
        ]);
      }

      // Base query with relationships
      $query = FeeChallans::with(['student', 'class', 'section', 'staff', 'school', 'latestHistory.staff', 'history'])
        ->where('app_school_id', $schoolId)
        ->select([
          'id',
          'challan_no',
          'amount',
          'feemonth',
          'transid',
          'status',
          'app_student_id',
          'app_class_id',
          'app_section_id',
          'app_staff_id',
          'created_at'
        ]);

      // Get total count before filtering
      $recordsTotal = FeeChallans::where('app_school_id', $schoolId)->count();

      // Apply search filter
      $search = $request->input('search.value');
      if (!empty($search)) {
        $query->where(function ($q) use ($search) {
          $q->where('challan_no', 'like', "%{$search}%")
            ->orWhere('amount', 'like', "%{$search}%")
            ->orWhere('feemonth', 'like', "%{$search}%")
            ->orWhere('transid', 'like', "%{$search}%")
            ->orWhere('status', 'like', "%{$search}%")
            ->orWhereHas('student', function ($studentQuery) use ($search) {
              $studentQuery->where('name', 'like', "%{$search}%")
                ->orWhere('roll_no', 'like', "%{$search}%");
            })
            ->orWhereHas('class', function ($classQuery) use ($search) {
              $classQuery->where('name', 'like', "%{$search}%");
            });
        });
      }

      // Apply class filter
      $classId = $request->input('class_id');
      if (!empty($classId)) {
        $query->where('app_class_id', $classId);
      }

      // Apply section filter
      $sectionId = $request->input('section_id');
      if (!empty($sectionId)) {
        $query->where('app_section_id', $sectionId);
      }

      // Apply status filter
      $status = $request->input('status');
      if (!empty($status)) {
        $query->where('status', $status);
      }

      // Get filtered count
      $recordsFiltered = $query->count();

      // Define columns mapping (matching DataTable columns order)
      $columns = ['', 'id', 'challan_no', 'feemonth', 'amount', '', '', 'transid', 'status', 'created_at', ''];

      // Apply ordering
      $orderColumnIndex = $request->input('order.0.column', 2); // Default to column 2 (challan_no)
      $orderDir = $request->input('order.0.dir', 'desc');

      if (isset($columns[$orderColumnIndex]) && !empty($columns[$orderColumnIndex])) {
        $query->orderBy($columns[$orderColumnIndex], $orderDir);
      } else {
        $query->orderBy('challan_no', 'desc');
      }

      // Apply pagination
      $start = $request->input('start', 0);
      $length = $request->input('length', 10);
      $query->offset($start)->limit($length);

      // Get results
      $challans = $query->get();

      // Format data for DataTables
      $data = [];
      foreach ($challans as $challan) {
        // Status badge
        $statusBadge = '';
        switch ($challan->status) {
          case 0:
            $statusBadge = '<span class="badge rounded-pill bg-label-info">Issued</span>';
            break;
          case 1:
            $statusBadge = '<span class="badge rounded-pill bg-label-warning">Pending</span>';
            break;
          case 2:
            $statusBadge = '<span class="badge rounded-pill bg-label-primary">Waiting Approval</span>';
            break;
          case 3:
            $statusBadge = '<span class="badge rounded-pill bg-label-danger">Cancelled</span>';
            break;
          case 4:
            $statusBadge = '<span class="badge rounded-pill bg-label-success">Paid</span>';
            break;
          default:
            $statusBadge = '<span class="badge rounded-pill bg-label-secondary">Unknown</span>';
        }

        // Format image path
        $imagePath = $challan->student && $challan->student->image
          ? asset($challan->student->image)
          : asset('assets/img/avatars/default-avatar.png');

        // Get activity info (event count and last date)
        $activityInfo = '';
        $eventCount = $challan->history ? $challan->history->count() : 0;
        if ($challan->latestHistory) {
          $eventDate = date('d-m-Y', strtotime($challan->latestHistory->created_at));
          $activityInfo = '<small class="text-muted">' . $eventDate . '<br>Events = ' . $eventCount . '</small>';
        } else {
          $activityInfo = '<small class="text-muted">No activity</small>';
        }

        $data[] = [
          'id' => $challan->id,
          'encrypted_id' => IdEncryption::encrypt($challan->id),
          'DT_RowId' => 'row_' . $challan->id,
          'challan_no' => $challan->challan_no,
          'student_id' => $challan->app_student_id,
          'student_encrypted_id' => IdEncryption::encrypt($challan->app_student_id),
          'student_name' => $challan->student ? $challan->student->name : 'N/A',
          'student_roll' => $challan->student ? $challan->student->roll_no : 'N/A',
          'student_image' => $imagePath,
          'feemonth' => $challan->feemonth ? date('F Y', strtotime($challan->feemonth)) : 'N/A',
          'amount' => $challan->amount,
          'description' => $challan->description,
          'class_section' => ($challan->class ? $challan->class->name : 'N/A') . ' / ' . ($challan->section ? $challan->section->name : 'N/A'),
          'transid' => $challan->transid ?? 'N/A',
          'status' => $statusBadge,
          'created_date' => $challan->created_at ? date('d-m-Y', strtotime($challan->created_at)) : 'N/A',
          'activity' => $activityInfo,
          'challan_status_code' => $challan->status, // Add status code for permission checks
        ];
      }

      // Add user permissions to response
      $user = $request->user();
      $staffTypeId = $user->app_stype_id ?? null;
      $canCreate = $this->canCreateChallan();

      return response()->json([
        'draw' => intval($request->input('draw')),
        'recordsTotal' => $recordsTotal,
        'recordsFiltered' => $recordsFiltered,
        'data' => $data,
        'userPermissions' => [
          'canCreate' => $canCreate,
          'staffTypeId' => $staffTypeId
        ]
      ]);
    } catch (\Exception $e) {
      Log::error('challanListData error: ' . $e->getMessage());
      return response()->json([
        'draw' => intval($request->input('draw')),
        'recordsTotal' => 0,
        'recordsFiltered' => 0,
        'data' => [],
        'error' => $e->getMessage()
      ], 500);
    }
  }

  /**
   * Show challan creation form.
   */
  public function challanCreate()
  {
    try {
      // Check if user can create challans
      if (!$this->canCreateChallan()) {
        $user = Auth::user();
        $staffTypeId = $user->app_stype_id ?? null;

        $errorMsg = 'Access Denied: You do not have permission to create new fee challans.';
        if ($staffTypeId == 2) {
          $errorMsg = 'Access Restricted: As a Cashier, you can only view and edit existing challans (Issued/Pending status only). Creating new challans is not permitted for your role.';
        } elseif ($staffTypeId == 3) {
          $errorMsg = 'Access Restricted: As a Viewer, you have read-only access to fee records. Creating or editing challans is not permitted for your role.';
        }

        if (request()->ajax() || request()->wantsJson()) {
          return response()->json(['error' => $errorMsg], 403);
        }
        return redirect()->route('admin.student.managefee')->with('error', $errorMsg);
      }

      $schoolId = Auth::user()->app_school_id ?? null;

      if (!$schoolId) {
        return view('myapp.admin.challan_form', [
          'classes' => [],
          'students' => [],
          'nextChallanNo' => 'CH-000001'
        ]);
      }

      // Get next challan number
      $lastChallan = FeeChallans::where('app_school_id', $schoolId)
        ->orderBy('id', 'desc')
        ->first();

      $nextNumber = 1;
      if ($lastChallan && $lastChallan->challan_no) {
        $parts = explode('-', $lastChallan->challan_no);
        $nextNumber = intval(end($parts)) + 1;
      }
      $nextChallanNo = 'CH-' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);

      // Get classes and students
      $classes = App_class::where('app_school_id', $schoolId)->get();
      $students = App_student::with(['class', 'section'])
        ->where('app_school_id', $schoolId)
        ->where('status', 0)
        ->get();

      // Check if this is an AJAX request for modal
      if (request()->ajax() || request()->wantsJson()) {
        return view('myapp.admin.challan_form', compact('classes', 'students', 'nextChallanNo'));
      }

      return view('myapp.admin.challan_form', compact('classes', 'students', 'nextChallanNo'));
    } catch (\Exception $e) {
      Log::error('Error loading challan form: ' . $e->getMessage());
      return response()->json(['error' => 'Error loading form'], 500);
    }
  }

  /**
   * Store a new fee challan.
   */
  public function challanStore(Request $request)
  {
    try {
      // Check if user can create challans
      if (!$this->canCreateChallan()) {
        $user = Auth::user();
        $staffTypeId = $user->app_stype_id ?? null;

        $errorMsg = 'Access Denied: You do not have permission to create new fee challans.';
        if ($staffTypeId == 2) {
          $errorMsg = 'Access Restricted: Cashiers cannot create new challans. You can only mark existing Issued/Pending challans as "Waiting Approval".';
        } elseif ($staffTypeId == 3) {
          $errorMsg = 'Access Restricted: Viewers have read-only access. Creating or editing challans is not permitted.';
        }

        return response()->json(['error' => $errorMsg], 403);
      }

      $schoolId = Auth::user()->app_school_id;
      $staffId = Auth::user()->id;

      $data = $request->validate([
        'challan_no' => 'required|string|unique:app_fee',
        'app_student_id' => 'required|integer|exists:app_students,id',
        'app_class_id' => 'required|integer|exists:app_classes,id',
        'app_section_id' => 'required|integer|exists:app_sections,id',
        'feemonth' => 'required|date_format:Y-m',
        'status' => 'nullable|integer|in:0,1,2,3,4',
        'transid' => 'nullable|string|max:100',
        'use_custom_fee' => 'nullable',
        'custom_amount' => 'nullable|numeric|min:0',
        'description' => 'nullable|string|max:1000',
        'cancellation_reason' => 'nullable|string|max:1000',
        'payment_method' => 'nullable|string|in:cash,bank_transfer',
        'bank_reference' => 'nullable|string|max:100',
      ]);

      // Validate cancellation reason when status is cancelled
      if ($request->input('status') == 3 && empty($request->input('cancellation_reason'))) {
        return response()->json([
          'success' => false,
          'error' => 'Cancellation reason is required when marking challan as cancelled'
        ], 422);
      }

      // Additional validation: if custom fee is enabled, custom_amount is required
      if ($request->has('use_custom_fee') && $request->input('use_custom_fee') == 'on') {
        if (empty($request->input('custom_amount'))) {
          return response()->json([
            'success' => false,
            'error' => 'Custom amount is required when using custom fee'
          ], 422);
        }
      }

      // Convert YYYY-MM to Date (YYYY-MM-01) for storage
      $feeDate = $data['feemonth'] . '-01';

      // Check for duplicate challan (same student, class, section, and month)
      // Exclude cancelled challans to allow re-issuing
      $existingChallan = FeeChallans::where('app_student_id', $data['app_student_id'])
        ->where('app_class_id', $data['app_class_id'])
        ->where('app_section_id', $data['app_section_id'])
        ->where('feemonth', $feeDate)
        ->where('app_school_id', $schoolId)
        ->where('status', '!=', 3) // Exclude cancelled challans
        ->first();

      if ($existingChallan) {
        return response()->json([
          'success' => false,
          'error' => 'A challan already exists for this student in ' . date('F Y', strtotime($feeDate)) . '. Challan No: ' . $existingChallan->challan_no
        ], 422);
      }

      // Check if custom fee is being used
      if ($request->has('use_custom_fee') && $request->input('use_custom_fee') == 'on') {
        // Use custom amount directly
        $totalAmount = $request->input('custom_amount');
      } else {
        // Convert YYYY-MM to Month Name for Extra Charges lookup (assuming app_fee_extra uses month names)
        $monthName = date('F', strtotime($feeDate));

        // Get student and calculate amount based on student fee + extra charges
        $student = App_student::find($data['app_student_id']);
        if (!$student) {
          return response()->json(['error' => 'Student not found'], 404);
        }

        // Student's monthly fee
        $studentFee = $student->fee ?? 0;

        // Get extra charges for this student for this month
        $extraCharges = App_fee_extra::where('app_student_id', $data['app_student_id'])
          ->where('feemonth', $monthName)
          ->where('status', 'active')
          ->sum('amount');

        // Calculate total payable amount
        $totalAmount = $studentFee + $extraCharges;
      }

      // Auto-generate transaction ID when status is Paid
      if ($request->input('status') == 4) {
        if (empty($data['transid'])) {
          $data['transid'] = date('Ymd') . '-' . strtoupper(substr(uniqid(), -4));
        }
      }

      // Add school and staff IDs
      $data['app_school_id'] = $schoolId;
      $data['app_staff_id'] = $staffId;
      $data['amount'] = $totalAmount;
      $data['description'] = $request->input('description');
      $data['status'] = $data['status'] ?? 0;
      $data['feemonth'] = $feeDate; // Store as date
      $data['cancellation_reason'] = $request->input('cancellation_reason');
      $data['payment_method'] = $request->input('payment_method');
      $data['bank_reference'] = $request->input('bank_reference');

      Log::info('Creating challan with amount', [
        'use_custom_fee' => $request->has('use_custom_fee'),
        'custom_amount' => $request->input('custom_amount'),
        'calculated_amount' => $totalAmount,
        'final_amount' => $data['amount']
      ]);

      // Verify student belongs to this school
      $student = App_student::where('id', $data['app_student_id'])
        ->where('app_school_id', $schoolId)
        ->first();

      if (!$student) {
        return response()->json(['error' => 'Invalid student'], 403);
      }

      // Create challan
      $challan = FeeChallans::create($data);

      Log::info('Fee challan created', [
        'challan_id' => $challan->id,
        'challan_no' => $challan->challan_no,
        'student_id' => $challan->app_student_id
      ]);

      return response()->json([
        'success' => true,
        'message' => 'Fee challan created successfully',
        'challan_id' => $challan->id,
        'challan_no' => $challan->challan_no
      ]);
    } catch (\Illuminate\Validation\ValidationException $e) {
      return response()->json(['errors' => $e->errors()], 422);
    } catch (\Exception $e) {
      Log::error('Error creating challan: ' . $e->getMessage());
      return response()->json(['error' => $e->getMessage()], 500);
    }
  }

  /**
   * Show the form for editing a challan.
   */
  public function challanEdit($id)
  {
    try {
      $schoolId = Auth::user()->app_school_id ?? null;

      // Get challan with relationships
      $challan = FeeChallans::with(['student', 'class', 'section'])
        ->where('id', $id)
        ->where('app_school_id', $schoolId)
        ->firstOrFail();

      // Check if user can edit this challan
      if (!$this->canEditChallan($challan)) {
        $user = Auth::user();
        $staffTypeId = $user->app_stype_id ?? null;

        $errorMsg = 'Access Denied: You do not have permission to edit this challan.';

        if ($staffTypeId == 3) {
          $errorMsg = 'Access Restricted: Viewers have read-only access. Editing challans is not permitted for your role.';
        } elseif ($staffTypeId == 2) {
          if ($challan->status == 4) {
            $errorMsg = 'Edit Restricted: This challan is already marked as PAID and cannot be modified. Paid challans are locked for data integrity.';
          } elseif ($challan->status == 3) {
            $errorMsg = 'Edit Restricted: This challan is CANCELLED and cannot be modified. Only Issued or Pending challans can be edited.';
          } elseif ($challan->status == 2) {
            $errorMsg = 'Edit Restricted: This challan is in WAITING APPROVAL status and cannot be modified. Only Issued or Pending challans can be edited by Cashiers.';
          } else {
            $errorMsg = 'Access Restricted: As a Cashier, you can only edit challans with Issued or Pending status. You can mark them as "Waiting Approval" for admin review.';
          }
        }

        return response()->json(['error' => $errorMsg], 403);
      }

      // Calculate expected fee to determine if custom amount is being used
      $monthName = date('F', strtotime($challan->feemonth));
      $studentFee = $challan->student->fee ?? 0;
      $extraCharges = App_fee_extra::where('app_student_id', $challan->app_student_id)
        ->where('feemonth', $monthName)
        ->where('status', 'active')
        ->sum('amount');
      $expectedAmount = $studentFee + $extraCharges;

      // Check if challan has custom amount (differs from expected)
      $isCustomAmount = abs($challan->amount - $expectedAmount) > 0.01; // Allow for floating point precision

      // Get classes and students for dropdowns
      $classes = App_class::where('app_school_id', $schoolId)->get();
      $students = App_student::with(['class', 'section'])
        ->where('app_school_id', $schoolId)
        ->get();

      // Get allowed status transitions for current user
      $allowedStatuses = $this->getAllowedStatusTransitions($challan->status);
      $staffTypeId = Auth::user()->app_stype_id ?? null;

      return view('myapp.admin.challan_form', compact('challan', 'classes', 'students', 'isCustomAmount', 'allowedStatuses', 'staffTypeId'));
    } catch (\Exception $e) {
      Log::error('Error loading challan edit form: ' . $e->getMessage());
      return response()->json(['error' => 'Challan not found'], 404);
    }
  }

  /**
   * Update a challan.
   */
  public function challanUpdate(Request $request, $id)
  {
    try {
      $schoolId = Auth::user()->app_school_id;
      $staffId = Auth::user()->id;

      // Get existing challan
      $challan = FeeChallans::where('id', $id)
        ->where('app_school_id', $schoolId)
        ->firstOrFail();

      // Check if user can edit this challan
      if (!$this->canEditChallan($challan)) {
        $user = Auth::user();
        $staffTypeId = $user->app_stype_id ?? null;

        $errorMsg = 'Access Denied: You do not have permission to edit this challan.';
        if ($staffTypeId == 3) {
          $errorMsg = 'Access Restricted: Viewers have read-only access. Editing challans is not permitted for your role.';
        } elseif ($staffTypeId == 2) {
          $errorMsg = 'Edit Restricted: As a Cashier, you can only edit Issued or Pending challans and mark them as "Waiting Approval". This challan cannot be modified.';
        }

        return response()->json(['error' => $errorMsg], 403);
      }

      // Get staff type for validation
      $user = Auth::user();
      $staffTypeId = $user->app_stype_id ?? null;

      // Validate status change permissions
      $requestedStatus = $request->input('status');
      if ($staffTypeId == 2 && $requestedStatus != 2) {
        $statusLabels = ['Issued', 'Pending', 'Waiting Approval', 'Cancelled', 'Paid'];
        $requestedStatusLabel = $statusLabels[$requestedStatus] ?? 'Unknown';

        return response()->json([
          'success' => false,
          'error' => 'Status Change Restricted: As a Cashier, you can only change challan status to "Waiting Approval". You cannot mark challans as "' . $requestedStatusLabel . '". Please submit for approval instead.'
        ], 403);
      }

      $data = $request->validate([
        'app_student_id' => 'required|integer|exists:app_students,id',
        'app_class_id' => 'required|integer|exists:app_classes,id',
        'app_section_id' => 'required|integer|exists:app_sections,id',
        'feemonth' => 'required|date_format:Y-m',
        'status' => 'nullable|integer|in:0,1,2,3,4',
        'transid' => 'nullable|string|max:100',
        'use_custom_fee' => 'nullable',
        'custom_amount' => 'nullable|numeric|min:0',
        'description' => 'nullable|string|max:1000',
        'cancellation_reason' => 'nullable|string|max:1000',
        'payment_method' => 'nullable|string|in:cash,bank_transfer',
        'bank_reference' => 'nullable|string|max:100',
      ]);

      // Validate cancellation reason when status is cancelled
      if ($request->input('status') == 3 && empty($request->input('cancellation_reason'))) {
        return response()->json([
          'success' => false,
          'error' => 'Cancellation reason is required when marking challan as cancelled'
        ], 422);
      }

      // Validate payment method when status is paid
      if ($request->input('status') == 4) {
        if (empty($request->input('payment_method'))) {
          return response()->json([
            'success' => false,
            'error' => 'Payment method is required when marking challan as paid'
          ], 422);
        }

        if ($request->input('payment_method') == 'bank_transfer' && empty($request->input('bank_reference'))) {
          return response()->json([
            'success' => false,
            'error' => 'Bank transaction reference is required for bank transfer payments'
          ], 422);
        }
      }

      // Additional validation for custom fee
      if ($request->has('use_custom_fee') && $request->input('use_custom_fee') == 'on') {
        if (empty($request->input('custom_amount'))) {
          return response()->json([
            'success' => false,
            'error' => 'Custom amount is required when using custom fee'
          ], 422);
        }
      }

      // Convert YYYY-MM to Date
      $feeDate = $data['feemonth'] . '-01';

      // Check for duplicate (excluding current challan and cancelled challans)
      $existingChallan = FeeChallans::where('app_student_id', $data['app_student_id'])
        ->where('app_class_id', $data['app_class_id'])
        ->where('app_section_id', $data['app_section_id'])
        ->where('feemonth', $feeDate)
        ->where('app_school_id', $schoolId)
        ->where('id', '!=', $id)
        ->where('status', '!=', 3) // Exclude cancelled challans
        ->first();

      if ($existingChallan) {
        return response()->json([
          'success' => false,
          'error' => 'A challan already exists for this student in ' . date('F Y', strtotime($feeDate)) . '. Challan No: ' . $existingChallan->challan_no
        ], 422);
      }

      // Calculate or use custom amount
      if ($request->has('use_custom_fee') && $request->input('use_custom_fee') == 'on') {
        $totalAmount = $request->input('custom_amount');
      } else {
        $monthName = date('F', strtotime($feeDate));
        $student = App_student::find($data['app_student_id']);
        if (!$student) {
          return response()->json(['error' => 'Student not found'], 404);
        }

        $studentFee = $student->fee ?? 0;
        $extraCharges = App_fee_extra::where('app_student_id', $data['app_student_id'])
          ->where('feemonth', $monthName)
          ->where('status', 'active')
          ->sum('amount');

        $totalAmount = $studentFee + $extraCharges;
      }

      // Auto-generate transaction ID when status is Waiting Approval or Paid
      if (in_array($request->input('status'), [2, 4])) {
        if (empty($data['transid'])) {
          $data['transid'] = date('Ymd') . '-' . strtoupper(substr(uniqid(), -4));
        }
      }

      // Track status change
      $oldStatus = $challan->status;
      $newStatus = $data['status'] ?? 0;

      // Update challan
      $challan->update([
        'app_student_id' => $data['app_student_id'],
        'app_class_id' => $data['app_class_id'],
        'app_section_id' => $data['app_section_id'],
        'feemonth' => $feeDate,
        'amount' => $totalAmount,
        'description' => $request->input('description'),
        'status' => $newStatus,
        'transid' => $data['transid'] ?? null,
        'cancellation_reason' => $request->input('cancellation_reason'),
        'payment_method' => $request->input('payment_method'),
        'bank_reference' => $request->input('bank_reference'),
        'app_staff_id' => $staffId,
      ]);

      // Log status change if status changed
      if ($oldStatus != $newStatus) {
        $notes = $request->input('cancellation_reason') ?? 'Status changed from ' . $this->getStatusLabel($oldStatus) . ' to ' . $this->getStatusLabel($newStatus);

        // Add special note for cashier marking as waiting approval
        if ($staffTypeId == 2 && $newStatus == 2) {
          $notes = 'Marked as Waiting Approval by Cashier (Staff ID: ' . $staffId . '). ' . $notes;
        }

        \App\Models\App_fee_history::create([
          'app_fee_id' => $challan->id,
          'old_status' => $oldStatus,
          'new_status' => $newStatus,
          'changed_by' => $staffId,
          'notes' => $notes
        ]);
      }

      Log::info('Fee challan updated', [
        'challan_id' => $challan->id,
        'challan_no' => $challan->challan_no
      ]);

      return response()->json([
        'success' => true,
        'message' => 'Fee challan updated successfully',
        'challan_id' => $challan->id,
        'challan_no' => $challan->challan_no
      ]);
    } catch (\Illuminate\Validation\ValidationException $e) {
      return response()->json(['errors' => $e->errors()], 422);
    } catch (\Exception $e) {
      Log::error('Error updating challan: ' . $e->getMessage());
      return response()->json(['error' => $e->getMessage()], 500);
    }
  }

  /**
   * Get extra charges for a student in a specific month
   */
  public function getFeeExtras(Request $request)
  {
    try {
      $studentId = $request->input('student_id');
      $month = $request->input('month'); // YYYY-MM

      // Convert YYYY-MM to Month Name
      $monthName = date('F', strtotime($month . '-01'));

      $total = App_fee_extra::where('app_student_id', $studentId)
        ->where('feemonth', $monthName)
        ->where('status', 'active')
        ->sum('amount');

      return response()->json([
        'total' => $total ?? 0,
        'student_id' => $studentId,
        'month' => $month
      ]);
    } catch (\Exception $e) {
      Log::error('Error fetching fee extras: ' . $e->getMessage());
      return response()->json(['total' => 0, 'error' => $e->getMessage()], 500);
    }
  }

  // ==================== STAFF MANAGEMENT ====================

  /**
   * Display staff list
   */
  public function staffList(Request $request)
  {
    $schoolId = Auth::user()->app_school_id;
    $stypes = App_stype::orderBy('name')->get();

    $query = App_staff::with(['stype', 'school'])
      ->where('app_school_id', $schoolId);

    // Filter by type
    if ($request->has('type_filter') && !empty($request->type_filter)) {
      $query->where('app_stype_id', $request->type_filter);
    }

    // Filter by status
    if ($request->has('status_filter') && $request->status_filter !== '') {
      $query->where('status', $request->status_filter);
    }

    // Search
    if ($request->has('search') && !empty($request->search)) {
      $searchValue = $request->search;
      $query->where(function ($q) use ($searchValue) {
        $q->where('name', 'like', "%{$searchValue}%")
          ->orWhere('email', 'like', "%{$searchValue}%")
          ->orWhere('mob', 'like', "%{$searchValue}%");
      });
    }

    $staff = $query->orderBy('name')->paginate(10);

    return view('myapp.admin.staff_list', compact('stypes', 'staff'));
  }

  /**
   * Get staff data for DataTables
   */
  public function staffListData(Request $request)
  {
    $schoolId = Auth::user()->app_school_id;

    $query = App_staff::with(['stype', 'school'])
      ->where('app_school_id', $schoolId);

    // Search
    if ($request->has('search') && !empty($request->search['value'])) {
      $searchValue = $request->search['value'];
      $query->where(function ($q) use ($searchValue) {
        $q->where('name', 'like', "%{$searchValue}%")
          ->orWhere('email', 'like', "%{$searchValue}%")
          ->orWhere('mob', 'like', "%{$searchValue}%");
      });
    }

    // Filter by type
    if ($request->has('type_filter') && !empty($request->type_filter)) {
      $query->where('app_stype_id', $request->type_filter);
    }

    // Filter by status
    if ($request->has('status_filter') && $request->status_filter !== '') {
      $query->where('status', $request->status_filter);
    }

    $totalRecords = App_staff::where('app_school_id', $schoolId)->count();
    $filteredRecords = $query->count();

    // Sorting
    if ($request->has('order')) {
      $orderColumn = $request->columns[$request->order[0]['column']]['data'];
      $orderDir = $request->order[0]['dir'];
      $query->orderBy($orderColumn, $orderDir);
    }

    // Pagination
    $staff = $query->skip($request->start ?? 0)
      ->take($request->length ?? 10)
      ->get();

    $data = $staff->map(function ($s) {
      return [
        'id' => $s->id,
        'encrypted_id' => $s->encrypted_id,
        'name' => $s->name,
        'email' => $s->email,
        'mob' => $s->mob,
        'type' => $s->stype->name ?? 'N/A',
        'type_id' => $s->app_stype_id,
        'status' => $s->status,
        'joined_at' => $s->joined_at ? date('d M Y', strtotime($s->joined_at)) : 'N/A',
      ];
    });

    return response()->json([
      'draw' => $request->draw,
      'recordsTotal' => $totalRecords,
      'recordsFiltered' => $filteredRecords,
      'data' => $data
    ]);
  }

  /**
   * Show staff creation form
   */
  public function staffCreate()
  {
    $stypes = App_stype::all();
    return view('myapp.admin.staff_form', compact('stypes'));
  }

  /**
   * Store new staff
   */
  public function staffStore(Request $request)
  {
    try {
      $schoolId = Auth::user()->app_school_id;

      $data = $request->validate([
        'name' => 'required|string|max:255',
        'mob' => 'required|string|max:20',
        'email' => 'required|email|unique:app_staff,email',
        'password' => 'required|string|min:6',
        'address' => 'nullable|string|max:500',
        'app_stype_id' => 'required|integer|exists:app_stypes,id',
        'status' => 'required|integer|in:0,1',
      ]);

      $data['password'] = bcrypt($data['password']);
      $data['app_school_id'] = $schoolId;
      $data['joined_at'] = now();

      $staff = App_staff::create($data);

      Log::info('Staff created', ['staff_id' => $staff->id, 'name' => $staff->name]);

      return response()->json([
        'success' => true,
        'message' => 'Staff member added successfully',
        'staff_id' => $staff->id
      ]);
    } catch (\Illuminate\Validation\ValidationException $e) {
      return response()->json(['errors' => $e->errors()], 422);
    } catch (\Exception $e) {
      Log::error('Error creating staff: ' . $e->getMessage());
      return response()->json(['error' => $e->getMessage()], 500);
    }
  }

  /**
   * Show staff edit form
   */
  public function staffEdit($id)
  {
    try {
      $schoolId = Auth::user()->app_school_id;

      // Decrypt ID if it's encrypted
      if (!is_numeric($id)) {
        $id = decrypt($id);
      }

      $staff = App_staff::where('id', $id)
        ->where('app_school_id', $schoolId)
        ->firstOrFail();

      $stypes = App_stype::all();

      return view('myapp.admin.staff_form', compact('staff', 'stypes'));
    } catch (\Exception $e) {
      Log::error('Error loading staff edit form: ' . $e->getMessage());
      return response()->json(['error' => 'Staff not found or invalid ID'], 404);
    }
  }

  /**
   * Update staff
   */
  public function staffUpdate(Request $request, $id)
  {
    try {
      $schoolId = Auth::user()->app_school_id;

      // Decrypt ID if it's encrypted
      if (!is_numeric($id)) {
        $id = decrypt($id);
      }

      $staff = App_staff::where('id', $id)
        ->where('app_school_id', $schoolId)
        ->firstOrFail();

      $data = $request->validate([
        'name' => 'required|string|max:255',
        'mob' => 'required|string|max:20',
        'email' => 'required|email|unique:app_staff,email,' . $id,
        'password' => 'nullable|string|min:6',
        'address' => 'nullable|string|max:500',
        'app_stype_id' => 'required|integer|exists:app_stypes,id',
        'status' => 'required|integer|in:0,1',
      ]);

      if (!empty($data['password'])) {
        $data['password'] = bcrypt($data['password']);
      } else {
        unset($data['password']);
      }

      $staff->update($data);

      Log::info('Staff updated', ['staff_id' => $staff->id, 'name' => $staff->name]);

      return response()->json([
        'success' => true,
        'message' => 'Staff member updated successfully',
        'staff_id' => $staff->id
      ]);
    } catch (\Illuminate\Validation\ValidationException $e) {
      return response()->json(['errors' => $e->errors()], 422);
    } catch (\Exception $e) {
      Log::error('Error updating staff: ' . $e->getMessage());
      return response()->json(['error' => $e->getMessage()], 500);
    }
  }
}
