<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ChainSelectController extends Controller
{
  /**
   * Fetch options for dependent select
   *
   * @param Request $request
   * @return \Illuminate\Http\JsonResponse
   */
  public function fetchOptions(Request $request)
  {
    $validator = Validator::make($request->all(), [
      'model' => 'required|string',
      'filterColumn' => 'required|string',
      'filterValue' => 'required',
      'display' => 'required|string',
      'key' => 'required|string',
    ]);

    if ($validator->fails()) {
      return response()->json([
        'success' => false,
        'message' => 'Validation failed',
        'errors' => $validator->errors()
      ], 422);
    }

    $model = $request->input('model');
    $filterColumn = $request->input('filterColumn');
    $filterValue = $request->input('filterValue');
    $display = $request->input('display');
    $key = $request->input('key');

    // Security: validate model class exists and is safe
    if (!class_exists($model)) {
      return response()->json([
        'success' => false,
        'message' => 'Invalid model'
      ], 400);
    }

    try {
      // Handle both single and multiple filter values
      $filterValue = $request->input('filterValue');

      // Build query
      $query = $model::query();

      // If filterValue is an array (multiple selection), use whereIn
      if (is_array($filterValue)) {
        $query->whereIn($filterColumn, $filterValue);
      } else {
        $query->where($filterColumn, $filterValue);
      }

      $items = $query->orderBy($display, 'asc')
        ->get([$key, $display])
        ->toArray();

      return response()->json([
        'success' => true,
        'items' => $items
      ]);
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => 'Error fetching data: ' . $e->getMessage()
      ], 500);
    }
  }
}
