<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class AjaxSearchController extends Controller
{
  /**
   * Search records from a model with AJAX
   *
   * @param Request $request
   * @return \Illuminate\Http\JsonResponse
   */
  public function search(Request $request)
  {
    $model = $request->input('model');
    $searchTerm = $request->input('search');
    $display = $request->input('display', 'name');
    $key = $request->input('key', 'id');
    $limit = $request->input('limit', 20);
    $page = $request->input('page', 1);

    // For chain selection support
    $filterColumn = $request->input('filterColumn');
    $filterValue = $request->input('filterValue');

    // Validate model exists and is safe
    if (!$model || !class_exists($model)) {
      return response()->json(['error' => 'Invalid model'], 400);
    }

    try {
      $query = $model::query();

      // AJAX search ALWAYS requires a search term (minimumInputLength: 1)
      // Even for chain selects, user must type to search
      if (!$searchTerm) {
        return response()->json([
          'results' => [],
          'pagination' => ['more' => false]
        ]);
      }

      // Apply parent filter if provided (for chain selection with AJAX)
      $hasParentFilter = $filterColumn && $filterValue && (!is_array($filterValue) || count($filterValue) > 0);

      if ($hasParentFilter) {
        if (is_array($filterValue)) {
          $query->whereIn($filterColumn, $filterValue);
        } else {
          $query->where($filterColumn, $filterValue);
        }
      }

      // Apply search term
      $query->where($display, 'LIKE', '%' . $searchTerm . '%');

      // Get total count before pagination
      $total = $query->count();

      // Apply pagination
      $results = $query->select($key, $display)
        ->orderBy($display, 'asc')
        ->limit($limit)
        ->offset(($page - 1) * $limit)
        ->get();

      // Format for Select2
      $items = $results->map(function ($item) use ($key, $display) {
        return [
          'id' => $item->{$key},
          'text' => $item->{$display}
        ];
      });

      return response()->json([
        'results' => $items,
        'pagination' => [
          'more' => ($page * $limit) < $total
        ]
      ]);
    } catch (\Exception $e) {
      return response()->json(['error' => 'Search failed: ' . $e->getMessage()], 500);
    }
  }
}
