<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Log;
use Illuminate\Contracts\Encryption\DecryptException;

class IdEncryption
{
    /**
     * Encrypt an ID for public use
     *
     * @param int|string $id
     * @return string|null
     */
    public static function encrypt($id)
    {
        if (empty($id)) {
            return null;
        }

        try {
            return Crypt::encryptString((string)$id);
        } catch (\Exception $e) {
            Log::error('ID Encryption Error: ' . $e->getMessage(), ['id' => $id]);
            return null;
        }
    }

    /**
     * Decrypt an ID from encrypted string
     *
     * @param string $encryptedId
     * @return int|null
     */
    public static function decrypt($encryptedId)
    {
        if (empty($encryptedId)) {
            return null;
        }

        try {
            $decrypted = Crypt::decryptString($encryptedId);
            return is_numeric($decrypted) ? (int)$decrypted : null;
        } catch (DecryptException $e) {
            Log::warning('ID Decryption Error: ' . $e->getMessage(), ['encrypted_id' => $encryptedId]);
            return null;
        }
    }

    /**
     * Encrypt multiple IDs
     *
     * @param array $ids
     * @return array
     */
    public static function encryptMany(array $ids)
    {
        return array_map(function ($id) {
            return self::encrypt($id);
        }, $ids);
    }

    /**
     * Decrypt multiple IDs
     *
     * @param array $encryptedIds
     * @return array
     */
    public static function decryptMany(array $encryptedIds)
    {
        return array_filter(array_map(function ($encryptedId) {
            return self::decrypt($encryptedId);
        }, $encryptedIds));
    }

    /**
     * Check if a string is a valid encrypted ID
     *
     * @param string $value
     * @return bool
     */
    public static function isEncrypted($value)
    {
        if (empty($value) || !is_string($value)) {
            return false;
        }

        try {
            $decrypted = Crypt::decryptString($value);
            return is_numeric($decrypted);
        } catch (DecryptException $e) {
            return false;
        }
    }
}
